package com.xebialabs.deployit.core.upgrade.configuration.common

import ai.digital.config.server.api.{ConfigurationPropertiesService, FileExtension, FilterCondition, PropertyValue}
import ai.digital.config.{ConfigFileFilter, ConfigFileMapping}
import com.typesafe.config.{Config, ConfigFactory, ConfigRenderOptions}
import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired

import java.io.File
import java.nio.file.Files
import java.util
import scala.jdk.CollectionConverters.SeqHasAsJava
import scala.util.{Failure, Success, Try}

trait BaseConfigurationUpgrader extends Upgrade with Logging {

  @Autowired
  var defaultConfigurationPropertiesService: ConfigurationPropertiesService = _

  @Autowired
  var configFileFilter: ConfigFileFilter = _

  var parentPath = ""

  override def doUpgrade(): Boolean = Try {
    updateConfiguration()
  } match {
    case Success(_) =>
      logger.info(s"$configurationFileDescription configuration has been written successfully.")
      true
    case Failure(exception: Exception) =>
      logger.error(s"$configurationFileDescription configuration cannot be written: " + exception.getMessage)
      false
    case _ =>
      false
  }

  def confFolder: String = s"${parentPath}conf"

  def centralConfFolder: String = s"${parentPath}centralConfiguration"

  def destinationFileName: String

  def destinationFile: File = new File(s"$centralConfFolder/$destinationFileName")

  def deployitConfFile: File = new File(s"$confFolder/deployit.conf")

  def xlDeployConfFile: File = new File(s"$confFolder/xl-deploy.conf")

  def configurationFileDescription: String

  override def upgradeVersion(): Version = Version.valueOf("central-config", "10.2.0")

  protected def updateConfiguration(): Unit

  def excludeXlDeployConfSections(paths: List[String]): Unit = {
    if (xlDeployConfFile.exists()) {
      def conf: Config = ConfigFactory.parseFile(xlDeployConfFile)

      paths.foreach { path =>
        if (conf.hasPath(path)) {
          Files.write(xlDeployConfFile.toPath,
            Seq(
              conf.withoutPath(path).root().
                render(ConfigRenderOptions.defaults.setFormatted(true).setJson(false).
                  setOriginComments(false).setComments(true))
            ).asJava)
        }
      }
    }
  }

  def storeCentralConfigurationProperties(pairs: Seq[(String, util.Map[String, AnyRef])]): Iterable[String] = {
    defaultConfigurationPropertiesService.propertiesStore(
      configFileFilter,
      "Updated from central configuration upgarder",
      pairs.map {
        case (key, value) => (key, PropertyValue(value))
      },
      ConfigFileMapping(destinationFileName)
    )
  }

  def updateCentralConfigurationProperties(properties: Map[String, AnyRef]): Iterable[String] = {
    defaultConfigurationPropertiesService.propertiesUpdate(
      configFileFilter,
      properties.view.mapValues(v => PropertyValue(v)).toMap,
      ConfigFileMapping(destinationFileName)
    )
  }

  def deleteCentralConfigurationProperties(properties: Seq[String]): Iterable[String] = {
    defaultConfigurationPropertiesService.propertiesDelete(
      configFileFilter,
      properties.toSet,
      ConfigFileMapping(destinationFileName)
    )
  }

  def encryptCentralConfigurationProperties(): Iterable[String] = {
    defaultConfigurationPropertiesService.propertiesEncrypt(
      configFileFilter,
      FileExtension.YAML,
      "password",
      FilterCondition.CONTAINS
    )
  }
}
