package com.xebialabs.deployit.core.upgrade.configuration

import ai.digital.configuration.central.deploy.ServerSideProperties
import com.typesafe.config.Config
import com.xebialabs.deployit.core.upgrade.configuration.common.{BaseConfigurationUpgrader, ConfigUtils}
import com.xebialabs.deployit.server.api.upgrade.Version

import scala.util.{Failure, Success, Try}

/*
Configuration Upgrader for 10.3 for left over configurations in xl-deploy.conf to be moved to deploy-server.yaml
*/
class Deploy1040ServerConfigurationUpgrader extends BaseConfigurationUpgrader {

  protected var xlDeployConfOption: Option[Config] = None

  override def destinationFileName: String = "deploy-server.yaml"

  override def configurationFileDescription: String = "Server config"

  override def upgradeVersion(): Version = Version.valueOf("central-config", "10.4.0")

  override def doUpgrade(): Boolean = Try {
    updateConfiguration()
  } match {
    case Success(_) =>
      logger.debug(s"$configurationFileDescription configuration has been updated successfully.")
      true
    case Failure(exception: Exception) =>
      logger.debug(s"$configurationFileDescription configuration cannot be updated: " + exception.getMessage)
      false
    case _ =>
      false
  }

  override protected def updateConfiguration(): Unit = {
    xlDeployConfOption = ConfigUtils.loadXlDeployConf(xlDeployConfFile)
    xlDeployConfOption.map(conf => {
      Try {
        // Read from xl-deploy and create map of configs
        val confToUpdate = getConfToUpdate(conf)
        updateCentralConfigurationProperties(confToUpdate)
        // clean up
        excludeXlDeployConfSections(List(XL_DEPLOY_UI_EXTENSION, XL_SERVER_EXTENSION))
        logger.debug(s"Moved $confToUpdate from xl-deploy.conf to deploy-server.yaml")
      } match {
        case Failure(e) =>
          logger.error(s"Failure in moving configuration form xl-deploy.conf to : $destinationFile", e)
          throw e
        case _ =>
      }
    })
  }

  /**
   * Read from xl-deploy.conf and create map for the specific set of confs to be updated to the destination file
   * @param deployConf - config from conf file
   * @return map of subset config from server config
   */
  def getConfToUpdate(deployConf: Config): Map[String, AnyRef] = {
    var confToUpdate = Map[String, AnyRef]()
    if (deployConf.hasPath(XL_DEPLOY_UI_EXTENSION_FILE)) {
      confToUpdate += (SERVER_EXTENSIONS_UI_FILE ->
        ConfigUtils.getValueOrDefault(deployConf, XL_DEPLOY_UI_EXTENSION_FILE, ServerSideProperties.DEFAULT_DEPLOY_UI_EXTENSION_FILE))
    }
    if (deployConf.hasPath(XL_SERVER_EXTENSIONS_FILE)) {
      confToUpdate += (SERVER_EXTENSIONS_FILE ->
        ConfigUtils.getValueOrDefault(deployConf, XL_SERVER_EXTENSIONS_FILE, ServerSideProperties.DEFAULT_DEPLOY_SERVER_EXTENSION_FILE))
    }
    if (deployConf.hasPath(XL_SERVER_EXTENSIONS_TIMEOUT)) {
      confToUpdate += (SERVER_EXTENSIONS_TIMEOUT ->
        ConfigUtils.getValueOrDefault(deployConf, XL_SERVER_EXTENSIONS_TIMEOUT, ServerSideProperties.DEFAULT_DEPLOY_SERVER_EXTENSION_TIMEOUT))
    }
    if (deployConf.hasPath(XL_SERVER_EXTENSIONS_ROOT_PATH)) {
      confToUpdate += (SERVER_EXTENSIONS_ROOT_PATH ->
        ConfigUtils.getValueOrDefault(deployConf, XL_SERVER_EXTENSIONS_ROOT_PATH, ServerSideProperties.DEFAULT_DEPLOY_SERVER_EXTENSION_ROOT_PATH))
    }
    if (deployConf.hasPath(XL_SERVER_EXTENSIONS_SCRIPT_PATH_PREFIX)) {
      confToUpdate += (SERVER_EXTENSIONS_SCRIPT_PATH_PREFIX ->
        ConfigUtils.getValueOrDefault(deployConf, XL_SERVER_EXTENSIONS_SCRIPT_PATH_PREFIX, ServerSideProperties.DEFAULT_DEPLOY_SERVER_EXTENSION_SCRIPT_PATH_PREFIX))
    }
    confToUpdate
  }

  val XL_SERVER_EXTENSION = "xl.server-extension"
  val XL_SERVER_EXTENSIONS_FILE = "xl.server-extension.file"
  val XL_SERVER_EXTENSIONS_TIMEOUT = "xl.server-extension.timeout"
  val XL_SERVER_EXTENSIONS_ROOT_PATH = "xl.server-extension.rootPath"
  val XL_SERVER_EXTENSIONS_SCRIPT_PATH_PREFIX = "xl.server-extension.scriptsPathPrefix"
  val SERVER_EXTENSIONS_FILE = "deploy.server.extensions.server.file"
  val SERVER_EXTENSIONS_TIMEOUT = "deploy.server.extensions.server.timeout"
  val SERVER_EXTENSIONS_ROOT_PATH = "deploy.server.extensions.server.rootPath"
  val SERVER_EXTENSIONS_SCRIPT_PATH_PREFIX = "deploy.server.extensions.server.scriptsPathPrefix"
  val XL_DEPLOY_UI_EXTENSION = "xl-deploy.ui-extension"
  val XL_DEPLOY_UI_EXTENSION_FILE = "xl-deploy.ui-extension.file"
  val SERVER_EXTENSIONS_UI_FILE = "deploy.server.extensions.ui.file"
}
