package com.xebialabs.deployit.core.upgrade.configuration.common

import com.fasterxml.jackson.databind.{DeserializationFeature, ObjectMapper}
import com.fasterxml.jackson.databind.deser.std.BooleanDeserializerWithYesNoSupport
import com.fasterxml.jackson.databind.json.JsonMapper
import com.typesafe.config._
import com.xebialabs.deployit.core.upgrade.upgrade.OutputTransformerFn

import java.io.File
import java.nio.file.{Files, StandardCopyOption}

trait DeployConfUpgrader extends BaseConfigurationUpgrader {

  private lazy val objectMapper = JsonMapper.builder().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)
    .addModule(BooleanDeserializerWithYesNoSupport.createModule())
    .build()

  override protected def updateConfiguration(): Unit = {
    if (xlDeployConfFile.exists()) {
      val xlDeployConfig = ConfigFactory.parseFile(xlDeployConfFile)
      xlConfigUpdate(
        xlDeployConfig
          .resolveWith(xlDeployConfig, ConfigResolveOptions.defaults().setAllowUnresolved(true))
          .resolveWith(ConfigFactory.defaultReference())
      )
    }

    if (getYamlFileNamesToMove.nonEmpty) {
      getYamlFileNamesToMove.foreach { yamlFileName =>
        def yamlFile = new File(s"$centralConfFolder/$yamlFileName")

        if (yamlFile.exists()) {
          Files.move(yamlFile.toPath, destinationFile.toPath, StandardCopyOption.REPLACE_EXISTING)
        }
      }
    }
  }

  // Expecting to have max 1 element
  protected def getYamlFileNamesToMove: List[String] = List()

  protected def xlConfigUpdate(confFile: Config): Unit

  def createYamlConfiguration[T](confFile: Config,
                                 confPrefix: String,
                                 yamlPrefix: String,
                                 clazz: Class[T],
                                 outputTransformer: OutputTransformerFn[T] = Predef.identity[T](_)): Unit = {

    info(s"Destination file location is: $destinationFile")
    storeCentralConfigurationProperties(Seq((yamlPrefix, ConfigUtils.configAsMap(outputTransformer(getYamlConfig(confFile, confPrefix, clazz))))))
    excludeXlDeployConfSections(List(confPrefix))
  }

  def getYamlConfig[T](confFile: Config,
                       confPath: String,
                       clazz: Class[T]): T = {
    val sectionConfiguration = confFile.getConfig(confPath).root().render(ConfigRenderOptions.concise())
    objectMapper.readValue(sectionConfiguration, clazz)
  }

  def getYamlConfig[T](confFile: Config,
                       mapToNewKeyPairs: Map[String, String],
                       confPath: String,
                       clazz: Class[T]): T = {
    var config = confFile.getConfig(confPath)

    for ((originalKey, newKey) <- mapToNewKeyPairs) {
      val value = config.getAnyRef(originalKey)
      config = config.withValue(newKey, ConfigValueFactory.fromAnyRef(value)).withoutPath(originalKey)
    }

    val sectionConfiguration = config.root().render(ConfigRenderOptions.concise())
    new ObjectMapper().readValue(sectionConfiguration, clazz)
  }

}
