package com.xebialabs.deployit.core.upgrade.configuration

import ai.digital.configuration.central.deploy.ServerSideProperties.KEY_ACTIVE_USER_SESSIONS
import ai.digital.configuration.central.deploy._
import com.fasterxml.jackson.annotation.{JsonIgnoreProperties, JsonProperty}
import com.typesafe.config.Config
import com.xebialabs.deployit.ServerConfiguration.KEY_SERVER_RESOLVE_APPLICATION_DEPENDENCIES
import com.xebialabs.deployit.core.upgrade.configuration.ServerSidePropertiesUpgrader._
import com.xebialabs.deployit.core.upgrade.configuration.common.{ComposedConfUpgrader, OidcUtils}

import java.beans.BeanProperty
import java.io.File
import java.util
import java.util.{Properties, List => JList}
import scala.language.reflectiveCalls

class ServerSidePropertiesUpgrader extends ComposedConfUpgrader[ServerSideProperties] {

  override def destinationFileName: String = "deploy-server.yaml"

  override def configurationFileDescription: String = "Server"

  override protected def readCentralConfigYamlFile(config: ServerSideProperties): ServerSideProperties = config

  override protected def getInitialConfig: ServerSideProperties =
    if (xlDeployConf.hasPath(SERVER_PATH)) {
      getYamlConfig(xlDeployConf, SERVER_PATH, classOf[ServerSideProperties])
    } else {
      new ServerSideProperties()
    }

  override protected def listXlDeployConfPrefixes: List[String] = List(

    ALGORITHMS_PATH, AKKA, DOWNLOADS_PATH, DEPLOY_LICENSE_PATH, PATCH_DICTIONARIES_PATH, SERVER_PATH

  )

  override protected def yamlPrefix: String = "deploy.server"

  override protected def listInsourcedYamlFiles: List[File] = List()

  override protected def processXlDeployConf(xlDeployConf: Config, config: ServerSideProperties): ServerSideProperties = {
    val serverConfig = config

    serverConfig.license = configureLicense()

    if (xlDeployConf.hasPath(DOWNLOADS_PATH)) {
      serverConfig.downloads = getYamlConfig(xlDeployConf, DOWNLOADS_PATH, classOf[Downloads])
    }

    if (xlDeployConf.hasPath(ALGORITHMS_PATH)) {
      serverConfig.algorithms = getYamlConfig(xlDeployConf, ALGORITHMS_PATH, classOf[Algorithms])
    }

    if (xlDeployConf.hasPath(PATCH_DICTIONARIES_PATH)) {
      serverConfig.patchDictionaries = getYamlConfig(
        xlDeployConf,
        Map("max-file-size" -> "max-file-size-mb"),
        PATCH_DICTIONARIES_PATH, classOf[PatchDictionaries])
    }

    if (OidcUtils.isEnabled(xlDeployConf, centralConfFolder)) {
      serverConfig.security.auth.provider = "oidc"
    }

    serverConfig
  }

  private def configureLicense(): License = {
    var lic = new License()

    if (xlDeployConf.hasPath(LICENSE_PATH)) {
      lic = getYamlConfig(xlDeployConf, LICENSE_PATH, classOf[License])
    }

    if (xlDeployConf.hasPath(DEPLOY_LICENSE_PATH)) {
      lic.daysBeforeWarning = xlDeployConf.getInt(s"$DEPLOY_LICENSE_PATH.daysBeforeWarning")
    }

    lic
  }

  override protected def additionalYamlConfigs(): Map[String, AnyRef] = {

    def getValue(key: String): String = {
      if (xlDeployConf.hasPath(key)) xlDeployConf.getString(key) else ""
    }

    def getList(key: String): JList[String] = {
      if (xlDeployConf.hasPath(key)) xlDeployConf.getStringList(key) else JList.of()
    }

    if (xlDeployConf != null && xlDeployConf.hasPath("akka")) {
      val config = new AkkaSlimConfig()
      config.io.dns.resolver = getValue("akka.io.dns.resolver")
      config.loggers = getList("akka.loggers")
      config.loglevel = getValue("akka.loglevel")
      Map("akka" -> config)
    } else {
      Map()
    }
  }

  override protected def listDeployitConfKeyNames: List[String] = List(
    KEY_SERVER_RESOLVE_APPLICATION_DEPENDENCIES,
    KEY_ACTIVE_USER_SESSIONS
  )

  override protected def processDeployItConf(properties: Properties, config: ServerSideProperties): ServerSideProperties = {
    config.resolveApplicationDependenciesEnabled = "true".equalsIgnoreCase(properties.getProperty(KEY_SERVER_RESOLVE_APPLICATION_DEPENDENCIES, "true"))
    config.session.activeUserSessionsEnabled = "true".equalsIgnoreCase(properties.getProperty(KEY_ACTIVE_USER_SESSIONS, "true"))
    config
  }

  override protected def processCentralConfigYamlFile(map: util.Map[String, AnyRef],
                                                      config: ServerSideProperties,
                                                      addIfExistsFn: AddIfExistsFn): ServerSideProperties = config
}

object ServerSidePropertiesUpgrader {
  val AKKA = "akka"
  val SERVER_PATH = "xl.server"
  val DEPLOY_LICENSE_PATH = "xl.deploy.license"
  val LICENSE_PATH = "xl.license"
  val DOWNLOADS_PATH = "xl.downloads"
  val ALGORITHMS_PATH = "xl.algorithms"
  val PATCH_DICTIONARIES_PATH = "xl.xldeploy.patch-dictionaries"
}

@JsonIgnoreProperties(ignoreUnknown = true)
class AkkaSlimConfig {

  @BeanProperty
  @JsonProperty
  var io = new {

    @BeanProperty
    @JsonProperty
    var dns = new {

      @BeanProperty
      @JsonProperty
      var resolver: String = _
    }
  }

  @BeanProperty
  @JsonProperty
  var loggers: JList[String] = JList.of()

  @BeanProperty
  @JsonProperty
  var loglevel: String = _
}
