package com.xebialabs.deployit.core.upgrade.configuration

import ai.digital.configuration.central.deploy.ClientProperties
import ai.digital.configuration.central.deploy.ClientProperties._

import java.io.File
import java.util
import java.util.Properties
import com.typesafe.config.Config
import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.core.upgrade.configuration.common.ComposedConfUpgrader
import com.xebialabs.deployit.util.PropertyUtil.parseIntSilent

import scala.jdk.CollectionConverters._
import scala.language.reflectiveCalls

class ClientPropertiesUpgrader extends ComposedConfUpgrader[ClientProperties] {

  type ClientPropertiesTransformerFn = (ClientProperties, String) => Unit

  override def configurationFileDescription: String = "Client"

  override def destinationFileName: String = "deploy-client.yaml"

  override protected def processDeployItConf(properties: Properties, config: ClientProperties): ClientProperties = {
    config.automaticallyMapAllDeployables = "true".equalsIgnoreCase(properties.getProperty(KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES, "true"))
    config.automaticallyMapAllDeployablesOnInitial = "true".equalsIgnoreCase(properties.getProperty(KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES_ON_INITIAL, "false"))
    config.automaticallyMapAllDeployablesOnUpdate = "true".equalsIgnoreCase(properties.getProperty(KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES_ON_UPDATE, "false"))
    config.sessionRememberEnabled = "true".equalsIgnoreCase(properties.getProperty(KEY_CLIENT_SESSION_REMEMBER_ENABLED, "true"))
    config.maintenance.forbiddenPaths = properties.getProperty(KEY_MAINTENANCE_FORBIDDEN_REQUESTS, ServerConfiguration.DEFAULT_MAINTENANCE_FORBIDDEN_REQUESTS).split(",").toList.asJava
    config.importablePackageDirectory = properties.getProperty(KEY_IMPORTABLE_PACKAGES_PATH, ServerConfiguration.DEFAULT_IMPORTABLE_PACKAGES_PATH)
    config.upgradeBatchSize = parseIntSilent(properties, KEY_UPGRADE_BATCH_SIZE, DEFAULT_UPGRADE_BATCH_SIZE)
    config.secureCookieEnabled = "true".equalsIgnoreCase(properties.getProperty(KEY_SECURE_COOKIE_ENABLED, "false"))
    config.uploadFileBufferSize = parseIntSilent(properties, KEY_UPLOAD_FILE_BUFFER_SIZE, DEFAULT_UPLOAD_FILE_BUFFER_SIZE)
    config.hideInternals = "true".equalsIgnoreCase(properties.getProperty(KEY_HIDE_INTERNALS, "false"))
    config.session.timeoutMinute = parseIntSilent(properties, KEY_CLIENT_SESSION_TIMEOUT_MINUTES, 20)
    config
  }

  override protected def processXlDeployConf(xlDeployConf: Config, config: ClientProperties): ClientProperties = {
    if (xlDeployConf.hasPath("xl.gui")) {
      if (xlDeployConf.hasPath("xl.gui.landing-page")) config.gui.landingPage = xlDeployConf.getString("xl.gui.landing-page") // TODO: VERIFY IT
      if (xlDeployConf.hasPath("xl.gui.task.status.poll.timeout-ms")) config.gui.task.status.poll.timeoutMs = xlDeployConf.getInt("xl.gui.task.status.poll.timeout-ms")
      if (xlDeployConf.hasPath("xl.gui.toastr.info.timeout-ms")) config.gui.toastr.info.timeoutMs = xlDeployConf.getInt("xl.gui.toastr.info.timeout-ms")
      if (xlDeployConf.hasPath("xl.gui.toastr.error.timeout-ms")) config.gui.toastr.error.timeoutMs = xlDeployConf.getInt("xl.gui.toastr.error.timeout-ms")
      if (xlDeployConf.hasPath("xl.gui.toastr.success.timeout-ms")) config.gui.toastr.success.timeoutMs = xlDeployConf.getInt("xl.gui.toastr.success.timeout-ms")
      if (xlDeployConf.hasPath("xl.gui.toastr.warning.timeout-ms")) config.gui.toastr.warning.timeoutMs = xlDeployConf.getInt("xl.gui.toastr.warning.timeout-ms")
    }
    config
  }

  override protected def processCentralConfigYamlFile(map: util.Map[String, AnyRef], config: ClientProperties, addIfExists: AddIfExistsFn): ClientProperties = {
    addIfExists(List("deploy.gui", "landing-page"), (config, value) => config.gui.landingPage = value)
    addIfExists(List("deploy.gui", "task", "status", "poll", "timeout-ms"), (config, value) => config.gui.task.status.poll.timeoutMs = Integer.valueOf(value))

    addIfExists(List("deploy.gui", "toastr", "info", "timeout-ms"), (config, value) => config.gui.toastr.info.timeoutMs = Integer.valueOf(value))
    addIfExists(List("deploy.gui", "toastr", "error", "timeout-ms"), (config, value) => config.gui.toastr.error.timeoutMs = Integer.valueOf(value))
    addIfExists(List("deploy.gui", "toastr", "success", "timeout-ms"), (config, value) => config.gui.toastr.success.timeoutMs = Integer.valueOf(value))
    addIfExists(List("deploy.gui", "toastr", "warning", "timeout-ms"), (config, value) => config.gui.toastr.warning.timeoutMs = Integer.valueOf(value))

    config
  }

  override protected def getInitialConfig: ClientProperties = new ClientProperties()

  override protected def yamlPrefix: String = "deploy.client"

  override protected def listInsourcedYamlFiles: List[File] = List(new File(s"$centralConfFolder/xld-client-gui.yaml"))

  override protected def listDeployitConfKeyNames: List[String] = List(
    KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES,
    KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES_ON_INITIAL,
    KEY_CLIENT_AUTOMATICALLY_MAP_ALL_DEPLOYABLES_ON_UPDATE,
    KEY_CLIENT_SESSION_REMEMBER_ENABLED,
    KEY_CLIENT_SESSION_TIMEOUT_MINUTES,
    KEY_MAINTENANCE_FORBIDDEN_REQUESTS,
    KEY_IMPORTABLE_PACKAGES_PATH,
    KEY_SECURE_COOKIE_ENABLED,
    KEY_UPGRADE_BATCH_SIZE,
    KEY_UPLOAD_FILE_BUFFER_SIZE,
    KEY_HIDE_INTERNALS
  )

  override protected def listXlDeployConfPrefixes: List[String] = List("xl.gui")

}
