package com.xebialabs.database.anonymizer

import org.kohsuke.args4j.{CmdLineException, CmdLineParser, Option}

import java.io.PrintStream

class AnonymizerOptions {

  @Option(name = "-f", usage = "Imports a specified data file", aliases = Array("-source"))
  var sourceFileName: String = _

  @Option(name = "-help", usage = "Prints this usage message")
  var printUsage: Boolean = false

  @Option(name = "-import", usage = "Imports data to empty database")
  var dbImport: Boolean = false

  @Option(name = "-refresh", usage = "Refreshes data in the database. (Every record will be verified prior to inserting. Therefore the import time increases)")
  var dbRefresh: Boolean = false

  @Option(name = "-batchSize", usage = "Specifies the maximum number of commands in a batch. Optimal batch size is different for each specific case and DBMS. However the default value 100 provides good enough results in the most cases. If you want to disable batch processing then set the value to '0'")
  var batchSize: Int = 100

  @Option(name = "-reports", usage = "Performs import/export on the reporting database")
  var reportingDb: Boolean = false
}

object AnonymizerOptions {
  private[anonymizer] def parse(args: String*): AnonymizerOptions = {
    val options = new AnonymizerOptions
    val parser = new CmdLineParser(options)
    try {
      parser.parseArgument(args: _*)
      if (options.printUsage) {
        printUsage(parser, System.err)
        System.exit(1)
      }
    } catch {
      case e: CmdLineException =>
        System.err.println(e.getMessage)
        printUsage(parser, System.out)
        System.exit(1)
    }
    options
  }

  private def printUsage(parser: CmdLineParser, stream: PrintStream): Unit = {
    stream.println("Usage: ./db-anonymizer." + (if (System.getProperty("os.name").matches("(?i).*windows.*")) "cmd"
    else "sh") + " [options] [[--] arguments]\n")
    stream.println("Options:\n")
    parser.printUsage(stream)
  }
}
