package ai.digital.deploy.metrics.repository.sql

import ai.digital.deploy.metrics.model.{CiCount, CiMetrics, FoldersCount, PluginMetrics, UsersCountByCi}
import ai.digital.deploy.metrics.repository.CiMetricsRepository
import ai.digital.deploy.metrics.repository.sql.SqlCiMetricsRepository.createCiMetricsRowMapper
import com.xebialabs.deployit.core.sql.spring.Setter
import com.xebialabs.deployit.core.sql.{OrderBy, Queries, SchemaInfo, SelectBuilder, SelectFragmentBuilder, SqlFunction, UTC_CALENDAR, toDateTime, SqlCondition => cond}
import com.xebialabs.deployit.engine.api.dto.{Ordering, Paging}
import com.xebialabs.deployit.repository.sql.HistorySchema
import com.xebialabs.deployit.sql.base.schema.CIS
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import java.sql.ResultSet
import scala.jdk.CollectionConverters.CollectionHasAsScala

@Component("ciMetricsRepositorySql")
@Transactional("mainTransactionManager")
class SqlCiMetricsRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                            (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends CiMetricsRepository with CiMetricsQueries {

   override def findCis(ciTypes: List[String], path: Option[String], createdBy: Option[String], createdAt: Option[String], paging: Paging, order: Ordering): List[CiMetrics] = {
    val builder = new SelectFragmentBuilder(SELECT_FRAGMENT_CI_METRICS).where(cond.in(CIS.ci_type, ciTypes))
    withPathPattern(builder, path)
    withUser(builder, createdBy)
    withDate(builder, createdAt)
    withPage(builder, paging)
    withOrder(builder, order)
    jdbcTemplate.query(builder.query, Setter(builder.parameters), ciMapper).asScala.toList
  }

  private def withPathPattern(builder: SelectBuilder, pathPattern: Option[String]) = pathPattern match {
    case Some(pattern) if pattern.nonEmpty =>
      builder.where(cond.likeEscaped(SqlFunction.lower(CIS.path), s"%$pattern%".toLowerCase()))
    case _ =>
  }

  private def withUser(builder: SelectBuilder, createdBy: Option[String]) = createdBy match {
    case Some(createdBy) if createdBy.nonEmpty =>
      builder.where(cond.equals(CIS.created_by, createdBy))
    case _ =>
  }

  private def withDate(builder: SelectBuilder, createdAt: Option[String]) = createdAt match {
    case Some(createdAt) if createdAt.nonEmpty =>
      val creationDate = DateTime.parse(createdAt)
      builder.where(cond.after(CIS.created_at, creationDate))
    case _ =>
  }

  private def withPlugins(builder: SelectBuilder, plugins: Option[List[String]], alias:Option[String]) = plugins match {
    case Some(plugins) if plugins != null && plugins.nonEmpty =>
      alias match {
        case Some(alias) if alias.nonEmpty =>
          builder.where(cond.in(CIS.ci_type.tableAlias(alias), plugins))
        case _ => builder.where(cond.in(CIS.ci_type, plugins))
      }

    case _ =>
  }

  private def withPage(builder: SelectBuilder, paging: Paging): Unit =
    Option(paging).filter(_.resultsPerPage != -1).foreach(p => builder.showPage(p.page, p.resultsPerPage))

  private def withOrder(builder: SelectBuilder, order: Ordering, alias: Option[String] = None) = {
    def orderingColumn(alias: Option[String]) = {
      SqlFunction.lower(alias.map(a => CIS.path.tableAlias(a)).getOrElse(CIS.path))
    }

    Option(order) match {
      case Some(ord) =>
        builder.orderBy(
          if (ord.isAscending) OrderBy.asc(orderingColumn(alias))
          else OrderBy.desc(orderingColumn(alias))
        )
      case _ =>
        builder.orderBy(OrderBy.asc(orderingColumn(alias)))
    }
  }

  private def ciMapper: RowMapper[CiMetrics] =
    createCiMetricsRowMapper()

  override def getCisCount(ciTypes: List[String], path: Option[String], createdBy: Option[String], createdAt: Option[String], plugins: Option[List[String]]): CiCount = {
    val selectBuilder = new SelectFragmentBuilder(SELECT_FRAGMENT_CIS_COUNT).where(cond.in(CIS.ci_type, ciTypes))
    withPathPattern(selectBuilder, path)
    withUser(selectBuilder, createdBy)
    withDate(selectBuilder, createdAt)
    jdbcTemplate.query(selectBuilder.query,
      Setter(selectBuilder.parameters),
      (rs: ResultSet, _: Int) => CiCount(path.getOrElse(""), rs.getInt(1))).asScala.headOption.getOrElse(CiCount(path.getOrElse(""), 0))
  }

  override def getSubFoldersCount(path: Option[String]): FoldersCount = {
    val selectBuilder = new SelectFragmentBuilder(SELECT_SUB_FOLDERS_COUNT).where(cond.equals(CIS.ci_type, "core.Directory"))
    withPathPattern(selectBuilder, path)
    jdbcTemplate.query(selectBuilder.query,
      Setter(selectBuilder.parameters),
    (rs: ResultSet, _: Int) => FoldersCount(path.getOrElse(""), rs.getLong(1))).asScala.headOption.getOrElse(FoldersCount(path.getOrElse(""), 0))}

  override def getUsersCount(ciTypes: List[String]): UsersCountByCi = {
    val selectBuilder = new SelectFragmentBuilder(SELECT_ACTIVE_USERS_COUNT).where(cond.in(CIS.ci_type.tableAlias("cis"), ciTypes))
    jdbcTemplate.query(selectBuilder.query,
        Setter(selectBuilder.parameters),
        (rs: ResultSet, _: Int) => UsersCountByCi(rs.getLong(1))).asScala.headOption.getOrElse(UsersCountByCi(0))
  }

  override def getPluginsDataByApplication(ciIds: List[Int], plugins:Option[List[String]]): List[PluginMetrics] = {
    getPluginsData(SELECT_PLUGINS_DATA_BY_APP, ciIds, plugins, "Applications")
  }

  override def getPluginsDataByEnvironment(ciIds: List[Int], plugins:Option[List[String]]): List[PluginMetrics] = {
    getPluginsData(SELECT_PLUGINS_DATA_BY_ENV, ciIds, plugins, "Environments")
  }

  override def getPluginsDataByInfra(ciIds: List[Int], plugins:Option[List[String]]): List[PluginMetrics] = {
    getPluginsData(SELECT_PLUGINS_DATA_BY_INFRA, ciIds, plugins, "Infrastructure")
  }

  private def getPluginsData(query: String, ciIds: List[Int], plugins:Option[List[String]], alias: String): List[PluginMetrics] = {
    val selectBuilder = new SelectFragmentBuilder(query).where(cond.in(CIS.ID.tableAlias(alias), ciIds))
    withPlugins(selectBuilder, plugins, Some("Plugins"))
    jdbcTemplate.query(selectBuilder.query,
      Setter(selectBuilder.parameters),
      new RowMapper[PluginMetrics] {
        override def mapRow(rs: ResultSet, rowNum: Int) = PluginMetrics(
          rs.getInt(CIS.ID.name),
          rs.getString(CIS.ci_type.name)
        )
      }).asScala.toList
  }

  override def getPluginsCountByApplication: Int = {
    val selectBuilder = new SelectFragmentBuilder(SELECT_PLUGINS_COUNT_BY_APP)
    jdbcTemplate.query(selectBuilder.query,
      (rs: ResultSet, _: Int) => rs.getInt(1)).asScala.headOption.getOrElse(0)
  }

  override def getPluginsCountByEnvironment: Int = {
    val selectBuilder = new SelectFragmentBuilder(SELECT_PLUGINS_COUNT_BY_ENV)
    jdbcTemplate.query(selectBuilder.query,
      (rs: ResultSet, _: Int) => rs.getInt(1)).asScala.headOption.getOrElse(0)
  }

  override def getPluginsCountByInfra(ciTypes : List[String]): Int = {
    val selectBuilder = new SelectFragmentBuilder(SELECT_PLUGINS_COUNT_BY_INFRA).where(cond.in(CIS.ci_type.tableAlias("Infrastructure"), ciTypes))
    jdbcTemplate.query(selectBuilder.query,
      Setter(selectBuilder.parameters),
      (rs: ResultSet, _: Int) => rs.getInt(1)).asScala.headOption.getOrElse(0)
  }

}

object SqlCiMetricsRepository {
  def createCiMetricsRowMapper()(implicit schemaInfo: SchemaInfo): RowMapper[CiMetrics] = (rs: ResultSet, _) =>  CiMetrics(
    id = rs.getInt(CIS.ID.name),
    name = rs.getString(CIS.name.name),
    ciType = rs.getString(CIS.ci_type.name),
    path = rs.getString(CIS.path.name),
    createdBy = rs.getString(CIS.created_by.name),
    createdAt =  Option(rs.getTimestamp(CIS.created_at.name, UTC_CALENDAR)).map(toDateTime).orNull,
    modifiedBy = rs.getString(CIS.modified_by.name),
    modifiedAt = Option(rs.getTimestamp(CIS.modified_at.name, UTC_CALENDAR)).map(toDateTime).orNull
  )
}

trait CiMetricsQueries extends Queries {
  import HistorySchema._

  lazy val SELECT_FRAGMENT_CIS_COUNT : String = sqlb"count(*) from ${CIS.tableName}"
  lazy val SELECT_FRAGMENT_CI_METRICS : String = sqlb"${CIS.ID}, ${CIS.name}, ${CIS.ci_type}, ${CIS.path}, ${CIS.created_by}, ${CIS.created_at}, ${CIS.modified_by}, ${CIS.modified_at} from ${CIS.tableName}"
  lazy val SELECT_ACTIVE_USERS_COUNT : String = sqlb"""count(distinct $changed_by) from $tableName history
    |INNER JOIN
    |${CIS.tableName} cis on history.$ci_id = cis.${CIS.ID}"""
  lazy val SELECT_SUB_FOLDERS_COUNT : String = sqlb"count(*) from ${CIS.tableName}"
  lazy val SELECT_PLUGINS_COUNT_BY_APP : String  = sqlb"""count(distinct Artifacts.${CIS.ci_type}) from ${CIS.tableName} Artifacts,
              |${CIS.tableName} Versions, ${CIS.tableName} Applications where
  |Artifacts.${CIS.parent_id} = Versions.${CIS.ID}
  |and Versions.${CIS.parent_id} = Applications.${CIS.ID}
  |and Applications.${CIS.ci_type} = 'udm.Application'"""
  lazy val SELECT_PLUGINS_DATA_BY_APP : String  = sqlb"""DISTINCT Applications.${CIS.ID},Plugins.${CIS.ci_type} from
   |${CIS.tableName} Plugins
   |INNER JOIN
   |${CIS.tableName} Versions on Plugins.${CIS.parent_id} = Versions.${CIS.ID}
   |INNER JOIN
   |${CIS.tableName} Applications on Versions.${CIS.parent_id} = Applications.${CIS.ID}
   |and Applications.${CIS.ci_type} = 'udm.Application'"""
  lazy val SELECT_PLUGINS_COUNT_BY_ENV: String = sqlb"""count(distinct Artifacts.${CIS.ci_type}) from
  |${CIS.tableName} Environments,
  |${CIS.tableName} DeployedApplications,
  |${CIS.tableName} Applications,
  |${CIS.tableName} Versions,
  |${CIS.tableName} Artifacts where
  |DeployedApplications.${CIS.parent_id} = Environments.${CIS.ID}
  |and DeployedApplications.${CIS.name}= Applications.${CIS.name}
  |and Versions.${CIS.parent_id} = Applications.${CIS.ID}
  |and Artifacts.${CIS.parent_id} = Versions.${CIS.ID}
  |and Applications.${CIS.ci_type} = 'udm.Application'
  |and DeployedApplications.${CIS.ci_type} = 'udm.DeployedApplication'
  |and Environments.${CIS.ci_type} = 'udm.Environment'"""
  lazy val SELECT_PLUGINS_DATA_BY_ENV: String =
    sqlb"""distinct Environments.${CIS.ID},
          |Plugins.${CIS.ci_type} from
          |${CIS.tableName} Environments
          |INNER JOIN
          |${CIS.tableName} DeployedApplications on DeployedApplications.${CIS.parent_id} = Environments.${CIS.ID}
          |and Environments.${CIS.ci_type} = 'udm.Environment'
          |INNER JOIN
          |${CIS.tableName} Applications on DeployedApplications.${CIS.name}= Applications.${CIS.name}
          |and Applications.${CIS.ci_type} = 'udm.Application'
          |and DeployedApplications.${CIS.ci_type} = 'udm.DeployedApplication'
          |INNER JOIN
          |${CIS.tableName} Versions on Versions.${CIS.parent_id} = Applications.${CIS.ID}
          |INNER JOIN
          |${CIS.tableName} Plugins on Plugins.${CIS.parent_id} = Versions.${CIS.ID}"""
  lazy val SELECT_PLUGINS_COUNT_BY_INFRA: String = sqlb"""count(distinct Containers.${CIS.ci_type})  from
          |${CIS.tableName} Containers
          |INNER JOIN
          |${CIS.tableName} Infrastructure ON
          |Containers.${CIS.parent_id} = Infrastructure.${CIS.ID}"""
  lazy val SELECT_PLUGINS_DATA_BY_INFRA: String = sqlb"""distinct Infrastructure.${CIS.ID}, Plugins.${CIS.ci_type}  from
  |${CIS.tableName} Plugins
  |INNER JOIN
  |${CIS.tableName} Infrastructure on Plugins.${CIS.parent_id} = Infrastructure.${CIS.ID}"""
}
