package com.xebialabs.gradle.plugins.thirdpartylicense.tasks

import groovy.json.JsonSlurper
import org.gradle.api.DefaultTask
import org.gradle.api.GradleException
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.ResolvedArtifact
import org.gradle.api.plugins.JavaBasePlugin
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.InputFile
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.Optional
import org.gradle.api.tasks.OutputFile
import org.gradle.api.tasks.TaskAction
import org.slf4j.Logger
import org.slf4j.LoggerFactory

import java.security.MessageDigest
import java.util.jar.JarFile
import java.util.zip.ZipEntry
import java.util.zip.ZipFile

class LicenseCheckAndDoc extends DefaultTask {
  private static final Logger logger = LoggerFactory.getLogger(LicenseCheckAndDoc.class)

  private File outputFile

  @InputFiles
  private Configuration configuration

  @InputFiles
  def licenseDatabaseFileOrConfiguration

  @Input
  def reportHeader

  @Input
  def ignoredGroups

  @Input
  private boolean failOnMissingLicense

  LicenseCheckAndDoc() {
    group = JavaBasePlugin.VERIFICATION_GROUP
  }

  def setLicenseDatabaseConfiguration(Configuration cfg) {
    licenseDatabaseFileOrConfiguration = cfg
  }

  def setLicenseDatabaseFile(file) {
    licenseDatabaseFileOrConfiguration = file
  }

  @Optional
  @InputFile
  File getLicenseDatabaseFile() {
    if (licenseDatabaseFileOrConfiguration instanceof Configuration) {
      return null
    }
    if (licenseDatabaseFileOrConfiguration != null) {
      return project.file(licenseDatabaseFileOrConfiguration)
    } else {
      return null
    }
  }

  @OutputFile
  File getOutputFile() {
    outputFile
  }

  void setOutputFile(String outputFile) {
    this.outputFile = project.file(outputFile)
  }

  void setOutputFile(File outputFile) {
    this.outputFile = outputFile
  }

  void setConfiguration(Configuration configuration) {
    this.configuration = configuration
    setDescription("Verify licenses and generate notice file with third party license information from ${configuration}")
  }

  Configuration getConfiguration() {
    return this.configuration
  }

  void setFailOnMissingLicense(boolean failOnMissingLicense) {
    this.failOnMissingLicense = failOnMissingLicense
  }

  Boolean getFailOnMissingLicense() {
    return this.failOnMissingLicense
  }

  @TaskAction
  def generateReport() {
    logger.info("Collecting license information for {}", configuration)

    def artifactLicenses = loadLicenseDatabase()

    getOutputFile().withWriter { out ->
      out.writeLine(reportHeader)

      def usedLicenses = [:]
      def missingLicenses = 0
      def updateKey = { String licenseKey, ResolvedArtifact artifact ->
        if (licenseKey != null && !licenseKey.isEmpty()) {
          if (usedLicenses.containsKey(licenseKey)) {
            logger.debug("Updating bucket for ${licenseKey} with artifact ${artifact.name}")
            usedLicenses[licenseKey].add(artifact)
          } else {
            logger.debug("Adding bucket for ${licenseKey} with artifact ${artifact.name}")
            usedLicenses[licenseKey] = [artifact] as Set
          }
        } else {
          println "Artifact ${artifact.name}: Missing license information (please fix license data)"
          missingLicenses++
        }
      }

      def unknownArtifacts = 0

      configuration.resolvedConfiguration.resolvedArtifacts.each { ResolvedArtifact artifact ->
        def version = artifact.moduleVersion.id.version
        def group = artifact.moduleVersion.id.group
        def name = artifact.name

        if (!(group in ignoredGroups)) {
          String artifactKey = "${group}:${name}"
          if (artifactLicenses.artifacts.containsKey(artifactKey)) {

            def source = artifactLicenses.artifacts[artifactKey].licenseSource
            switch (source) {
              case "data":
                updateKey(artifactLicenses.artifacts[artifactKey].license, artifact)
                break
              case "jar":
                def licenseData = readLicenseAndMd5(artifact.file)
                def licenseKey = licenseData.md5

                // abuse license data
                logger.debug("Adding license for $name with key $licenseKey")
                if (!artifactLicenses.licenses.containsKey(licenseKey)) {
                  artifactLicenses.licenses[licenseKey] = [:]
                  artifactLicenses.licenses[licenseKey].text = licenseData.text
                  artifactLicenses.licenses[licenseKey].type = artifactLicenses.artifacts[artifactKey].license
                }
                updateKey(licenseKey, artifact)
                break
            }
          } else {
            println "No license data specified for ${group}:${name}:${version}"
            unknownArtifacts++
          }
        }
      }

      if (failOnMissingLicense) {
        if (missingLicenses > 0) {
          throw new GradleException("Missing licenses in database: aborting.")
        }
        if (unknownArtifacts > 0) {
          throw new GradleException("Artifacts without license data encountered: aborting.")
        }
      }

      def sortedLicenses = []
      sortedLicenses.addAll(usedLicenses.keySet())
      sortedLicenses.sort()

      sortedLicenses.each { String key ->
        def sortedArtifacts = []
        sortedArtifacts.addAll(usedLicenses[key])
        sortedArtifacts.sort()

        if (!artifactLicenses.licenses.containsKey(key)) {
          throw new RuntimeException("missing key ${key}")
        }
        def licenseType = artifactLicenses.licenses[key].type
        String licenseFile = artifactLicenses.licenses[key].file
        String licenseText = artifactLicenses.licenses[key].text

        out.writeLine("Components using the ${licenseType} license:")

        sortedArtifacts.each { ResolvedArtifact artifact ->
          def additionalLicenses = artifactLicenses.artifacts["${artifact.moduleVersion.id.group}:${artifact.name}"].additionalLicenses
          def multiLicenseMsg = ""
          if (additionalLicenses != null) {
            multiLicenseMsg = "(Additionally licensed under " + additionalLicenses + ')'
          }
          out.writeLine(" * ${artifact.name}, version ${artifact.moduleVersion.id.version} ${multiLicenseMsg}")
        }

        if (licenseFile != null) {
          logger.debug("Going to read license: $licenseFile")
          licenseText = loadFileFromLicenseDatabaseOrLocal(licenseFile)
        }

        if (licenseText == null) {
          throw new GradleException("Null license text found for license key [${key}]")
        }

        // public domain has no text
        if (!licenseText.isEmpty()) {
          out.writeLine("\n\nThe full text of the license is:\n")
          out << licenseText
        }

        out.writeLine("\n\n\n\n")
      }
      out.writeLine("-- End of license notice. --\n")
    }
  }

  def loadLicenseDatabase() {
    try {
      String licenseDbText = loadFileFromLicenseDatabaseOrLocal("license-data.json")
      logger.debug("Loaded database: {}", licenseDbText)
      def artifactLicenses = new JsonSlurper().parseText(licenseDbText)
      checkJsonData(artifactLicenses)
      return artifactLicenses
    } catch (groovy.json.JsonException je) {
      throw new GradleException("Error reading license database: ${je.getMessage()}", je)
    }
  }

  String loadFileFromLicenseDatabaseOrLocal(String f) {
    String text
    if (getLicenseDatabaseFileOrConfiguration() != null) {
      def dbArtifact = getLicenseDatabaseFileOrConfiguration().resolvedConfiguration.resolvedArtifacts
      if (dbArtifact.size() != 1) {
        throw new IllegalArgumentException("Need exactly one artifact in " + licenseDatabaseFileOrConfiguration + " " + dbArtifact)
      }
      dbArtifact.each { artifact ->
        logger.info("Using license database from artifact {}:{}:{}:{}", artifact.name, artifact.moduleVersion.id.group, artifact.moduleVersion.id.version, artifact.file)
        def jar = new JarFile(artifact.file)
        logger.debug("Jar = {} --> getting {}", jar, f)
        ZipEntry ze = jar.getEntry(f)
        logger.debug("zipentry = {}", ze)
        if (ze == null) {
          throw new IllegalArgumentException("License database artifact did not contain $f")
        }
        text = jar.getInputStream(ze).getText()
      }
    } else {
      def file = project.file(f)
      logger.info("Using file {}", file)
      text = file.text
    }

    return text
  }

  def checkJsonData(def licenseData) {
    def problems = 0
    licenseData.artifacts.each { entry ->
      def artifact = entry.key
      def data = entry.value

      if (!data.containsKey('license')) {
        println "Validating license information: $artifact license key missing."
        problems++
      }
      if (!data.containsKey('licenseSource')) {
        println "Validating license information: $artifact licenseSource key missing."
        problems++
      } else if (!(data.licenseSource in ['jar', 'data'])) {
        println "Validating license information: $artifact licenseSource must be 'jar' or 'data' but was ${data.licenseSource}"
        problems++
      }

      def licenseKey = data.containsKey('license') ? data.license : "<missing-license-key>"

      if (data.licenseSource == 'data' && !licenseData.licenses.containsKey(licenseKey)) {
        println "Validating license information: $artifact refers to missing license $licenseKey"
        problems++
      }
    }
  }

  def md5Sum(String text) {
    byte[] data = text.getBytes()

    MessageDigest m = MessageDigest.getInstance("MD5")
    m.update(data, 0, data.length)

    BigInteger i = new BigInteger(1, m.digest())
    return String.format('%1$032X', i).toLowerCase()
  }

  def readLicenseAndMd5(File jarFile) {
    logger.debug("Reading license from ${jarFile}")
    def zip = new ZipFile(jarFile)
    def retVal = [:]
    try {
      zip.entries().each { ZipEntry e ->
        if (e.name =~ /([Ll]icense|LICENSE|GPL|LGPL|COPYING)(.txt)?$/) {
          logger.debug("Found license at ${e.name}")
          retVal.text = new String(zip.getInputStream(e).getText())
          // strip out whitespace before creating md5
          retVal.md5 = md5Sum(retVal.text.replaceAll(/\s+/, ''))
        }
      }
    }
    finally {
      zip.close()
    }
    if (retVal.isEmpty()) {
      throw new RuntimeException("No license found in ${jarFile}")
    }
    return retVal
  }
}
