package com.xebialabs.gradle.plugins.thirdpartylicense.tasks

import groovy.json.JsonBuilder
import org.gradle.api.DefaultTask
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.ResolvedArtifact
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.OutputFile
import org.gradle.api.tasks.TaskAction

import java.util.jar.JarFile
import java.util.jar.Manifest
import java.util.zip.ZipEntry

class LicenseReportTask extends DefaultTask {
  @OutputFile
  def outputFile

  @InputFiles
  Configuration configuration

  LicenseReportTask() {
    group = "reporting"
  }

  String getDescription() {
    return "Generate JSON report with third party license information from ${configuration}."
  }

  File getOutputFile() {
    project.file(outputFile)
  }

  @TaskAction
  def guessLicenses() {
    def artifactLicenses = [:]
    def licenses = [:]

    configuration.resolvedConfiguration.resolvedArtifacts.each { ResolvedArtifact artifact ->
      def version = artifact.moduleVersion.id.version
      def group = artifact.moduleVersion.id.group
      def name = artifact.name

      def guessed = guessLicense(artifact.file)

      def source = !guessed.jar.isEmpty() ? "jar" : "data"
      def detected = []
      detected.addAll(guessed.jar)
      detected.addAll(guessed.metadata)

      if (detected[0] != null && !guessed.text.isEmpty()) {
        licenses[detected[0]] = guessed.text
      }

      detected = detected.join(', ')

      String artifactKey = "${group}:${name}"
      artifactLicenses[artifactKey] = [license: detected, licenseSource: source]
    }

    def json = new JsonBuilder()
    json([artifacts: artifactLicenses, 'licenses': licenses])

    getOutputFile().withWriter { out ->
      out.write(json.toPrettyString())
    }
  }

  def guessLicense(File jarFile) {
    def licenseData = [:]
    def jar = new JarFile(jarFile)
    licenseData['metadata'] = getBundleLicenses(jar)
    licenseData['jar'] = []
    licenseData['text'] = []

    jar.entries().each { ZipEntry e ->
      if (e.name =~ /(license|LICENSE|GPL|LGPL|COPYING)(.txt)?$/) {
        def text = jar.getInputStream(e).getText()
        licenseData.text += text
        if (text =~ /Apache\s+License\s+Version\s+2.0/) {
          licenseData['jar'] += 'Apache-2.0'
        }
        if (text =~ /COMMON DEVELOPMENT AND DISTRIBUTION LICENSE (CDDL) Version 1.0/) {
          licenseData['jar'] += 'CDDL-1.0'
        }
        if (text =~ /COMMON DEVELOPMENT AND DISTRIBUTION LICENSE (CDDL) Version 1.1/) {
          licenseData['jar'] += 'CDDL-1.1'
        }
        if (text =~ /GNU\s+LIBRARY\s+GENERAL\s+PUBLIC\s+LICENSE\s+Version+s+2,/) {
          licenseData['jar'] += 'LGPL-2'
        }
        if (text =~ /GNU\s+LIBRARY\s+GENERAL\s+PUBLIC\s+LICENSE\s+Version+s+2.1,/) {
          licenseData['jar'] += 'LGPL-2.1'
        }
        if (text =~ /The GNU General Public License (GPL) Version 2,/) {
          licenseData['jar'] += 'GPL-2'
        }
      }
    }
    return licenseData
  }

  def getBundleLicenses(JarFile f) {
    Manifest m = f.getManifest()
    def entry = f.manifest?.mainAttributes.getValue("Bundle-License")
    if (entry == null) {
      return []
    }
    return entry.split(/\s*,\s*/)
  }
}
