package com.xebialabs.xlplatform.extensions.maven

import java.io.{FileInputStream, InputStream}
import java.net.URI

import com.xebialabs.deployit.engine.spi.artifact.resolution.ArtifactResolver.Resolver
import com.xebialabs.deployit.engine.spi.artifact.resolution.{ArtifactResolver, ResolvedArtifactFile}
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import org.apache.maven.repository.internal.MavenRepositorySystemUtils
import org.eclipse.aether.RepositorySystem
import org.eclipse.aether.artifact.DefaultArtifact
import org.eclipse.aether.connector.basic.BasicRepositoryConnectorFactory
import org.eclipse.aether.repository.LocalRepository
import org.eclipse.aether.resolution.ArtifactRequest
import org.eclipse.aether.spi.connector.RepositoryConnectorFactory
import org.eclipse.aether.spi.connector.transport.TransporterFactory
import org.eclipse.aether.transport.file.FileTransporterFactory
import org.eclipse.aether.transport.http.HttpTransporterFactory

import scala.collection.JavaConversions._
import scala.util.{Failure, Success, Try}

@Resolver(protocols = Array("maven"))
class MavenArtifactResolver extends ArtifactResolver {

  def getFileUri(artifact: SourceArtifact) = URI.create(artifact.getFileUri)

  override def resolveLocation(artifact: SourceArtifact): ResolvedArtifactFile = {

    val session = MavenRepositorySystemUtils.newSession()
    val workDir = getWorkDirFactory.newWorkDir("maven")
    Try {

      val localRepo = new LocalRepository(workDir.getPath)
      session.setLocalRepositoryManager(system.newLocalRepositoryManager(session, localRepo))

      val uri: URI = getFileUri(artifact)
      val request = createRequest(uri.getSchemeSpecificPart)

      val artifactResult = system.resolveArtifact(session, request)
      if (artifactResult.isMissing) {
        throw new IllegalArgumentException(
          s"Could not find maven artifact [${uri.getSchemeSpecificPart}}] in any of configured repositories")
      }
      val atf = artifactResult.getArtifact

      new ResolvedArtifactFile {
        override def getFileName: String = atf.getFile.getName
        override def openStream: InputStream = new FileInputStream(atf.getFile)
        override def close(): Unit = workDir.delete()
      }

    } match {
      case Failure(e: RuntimeException) =>
        workDir.delete()
        throw e
      case Failure(e) =>
        workDir.delete()
        throw new RuntimeException(e)
      case Success(resolved) => resolved
    }
  }

  override def validateCorrectness(artifact: SourceArtifact): Boolean = getFileUri(artifact).getScheme.equals("maven") &&
    (Try(new DefaultArtifact(getFileUri(artifact).getSchemeSpecificPart)) match {
      case Success(_) => true
      case Failure(e: IllegalArgumentException) => false
      case Failure(e @ _) => throw e
    })

  private def configureRepositorySystem(): RepositorySystem = {
    val locator = MavenRepositorySystemUtils.newServiceLocator()
      .addService(classOf[RepositoryConnectorFactory], classOf[BasicRepositoryConnectorFactory])
      .addService(classOf[TransporterFactory], classOf[FileTransporterFactory])
      .addService(classOf[TransporterFactory], classOf[HttpTransporterFactory])

    locator.getService(classOf[RepositorySystem])
  }

  private def createRequest(artifactCoordinates: String): ArtifactRequest = {
    val request = new ArtifactRequest()
    request.setArtifact(new DefaultArtifact(artifactCoordinates))
    request.setRepositories(getRepositories)
  }

  private lazy val system: RepositorySystem = configureRepositorySystem()
  private[maven] def getWorkDirFactory = MavenSettings.workDirFactory
  private[maven] def getRepositories = MavenSettings.repositories

}
