package com.xebialabs.satellite.streaming

import java.security.MessageDigest
import javax.xml.bind.annotation.adapters.HexBinaryAdapter

import akka.stream.scaladsl.Flow
import akka.stream.stage.{Context, PushStage, SyncDirective, TerminationDirective}
import akka.util.ByteString

object DigesterStage {
  type ProcessDigest = Digest => Unit

  def flow(onComplete: ProcessDigest) = Flow[ByteString].transform(() => new StreamDigester(onComplete))

  private class StreamDigester(onComplete: ProcessDigest) extends PushStage[ByteString, ByteString] {

    private val digester = new Digester()

    override def onPush(elem: ByteString, ctx: Context[ByteString]): SyncDirective = {
      digester.update(elem.toArray)
      ctx.push(elem)
    }

    override def onUpstreamFinish(ctx: Context[ByteString]): TerminationDirective = {
      onComplete(digester.digest())
      super.onUpstreamFinish(ctx)
    }

  }
  
  case class Digest(checksum: String, size: Long)
  
  private[streaming] class Digester {
    private val adapter = new HexBinaryAdapter()
    private val digester = MessageDigest.getInstance("SHA-1")
    var size = 0L

    def update(bytes: Array[Byte]): Unit = {
      size += bytes.length
      digester.update(bytes)
    }
    
    def digest(): Digest = Digest(adapter.marshal(digester.digest()), size)
  }
}

