package com.xebialabs.deployit.plugin.satellite

import java.io.ObjectInputStream
import java.net.URI

import akka.actor.ActorSystem
import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.deployit.plugin.satellite.extension.{ExtensionsLocator, FileSystemExtensionsLocator}
import com.xebialabs.satellite.future.AwaitForever
import com.xebialabs.satellite.protocol.Directories
import com.xebialabs.xlplatform.satellite.Satellite
import com.xebialabs.xlplatform.settings.CommonSettings

import scala.beans.BeanProperty

case class CheckExtensionsStep(actorLocator: ActorLocator, @BeanProperty description: String, extensionsLocator: ExtensionsLocator)
                          (implicit @transient var satelliteCommunicatorSystem: ActorSystem) extends Step with AwaitForever with SatellitePluginsSugar {

  @BeanProperty val order = Step.DEFAULT_ORDER

  override def execute(ctx: ExecutionContext): StepExitCode = {
    ctx.logOutput(s"Connecting to $actorLocator")
    if (!CommonSettings(satelliteCommunicatorSystem).satellite.enabled) {
      ctx.logError("Satellite has not been enabled on the XL Deploy server. Please ensure 'satellite.enabled = true' is set in the system.conf file of XL Deploy")
      StepExitCode.FAIL
    } else {

      val installedExtensions: InstalledExtensionsDelta = calculateExtensionsDelta(extensionsLocator, ctx)

      printStatus(installedExtensions.plugins, Directories.PLUGINS,  ctx)
      printStatus(installedExtensions.pluginHotfix, Directories.PLUGIN_HOTFIXES, ctx)
      printStatus(installedExtensions.satelliteHotfix, Directories.SATELLITE_HOTFIXES, ctx)
      printStatus(installedExtensions.ext, Directories.EXT, ctx)

      if (installedExtensions.empty) {
        ctx.logOutput(s"Satellite is up-to-date")
        StepExitCode.SUCCESS
      } else {
        ctx.logOutput(s"Found unsynchronized extensions")
        StepExitCode.FAIL
      }
    }
  }

  private def printStatus(opt: Set[URI], extension: String, ctx: ExecutionContext) = {
    if(opt.isEmpty) {
      ctx.logOutput(s"$extension is up-to-date")
    } else {
      ctx.logOutput(s"Need to synchronise $extension")
    }
  }

  private def readObject(in: ObjectInputStream): Unit = {
    in.defaultReadObject()
    satelliteCommunicatorSystem = SatelliteCommunicatorSystem.actorSystem
  }
}

object CheckExtensionsStep {

  def apply(satellite: Satellite)(implicit satelliteCommunicatorSystem: ActorSystem) : CheckExtensionsStep = {
    new CheckExtensionsStep(
      ActorLocator(satellite), s"Check plugins and extensions on satellite ${satellite.getName}",
      FileSystemExtensionsLocator
    )
  }

}
