package com.xebialabs.deployit.plugin.satellite

import akka.actor._
import akka.util.Timeout
import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.deployit.plugin.satellite.Pinger.{Unavailable, Up}
import com.xebialabs.satellite.protocol.{Paths, Ping, PingReply}
import com.xebialabs.xlplatform.satellite.Satellite
import com.xebialabs.xlplatform.settings.CommonSettings

import scala.concurrent.Await
import scala.concurrent.duration.FiniteDuration

object Pinger {

  val FIVE_PINGS = 5

  def defaultPingTimeout(satelliteCommunicatorSystem: ActorSystem) = CommonSettings(satelliteCommunicatorSystem).satellite.pingTimeout.duration

  def isUp(satellite: Satellite, noPings: Int = FIVE_PINGS)(implicit ctx: ExecutionContext, satelliteCommunicatorSystem: ActorSystem): Boolean = isUp(SatelliteAddress(satellite), ActorLocator(satellite), defaultPingTimeout(satelliteCommunicatorSystem), noPings, Clock())

  def isUp(satelliteAddress: SatelliteAddress, actorLocator: ActorLocator, pingTimeout: FiniteDuration, noPings: Int, clock: Clock)(implicit ctx: ExecutionContext, satelliteCommunicatorSystem: ActorSystem): Boolean = {
    import akka.pattern._

    implicit val timeout: Timeout = Timeout((pingTimeout * 1.5).asInstanceOf[FiniteDuration])
    implicit val dispatcher = satelliteCommunicatorSystem.dispatcher

    ctx.logOutput(s"Connecting to satellite at $satelliteAddress")

    val pinger = satelliteCommunicatorSystem.actorOf(Pinger.props(ctx, pingTimeout, satelliteAddress, clock, noPings))
    Await.result((pinger ? Pinger.Start(actorLocator.locate(Paths.ping)))
      .mapTo[PingResult]
      .recover { case error =>
        ctx.logError(s"Operation failed  (${error.getMessage})")
        Unavailable
      }.map(Up == _), timeout.duration)
  }

  def props(ctx: ExecutionContext, timeout: FiniteDuration, satelliteAddress: SatelliteAddress, clock: Clock = Clock(), noPings: Int = Pinger.FIVE_PINGS) = Props(classOf[Pinger], ctx, timeout, satelliteAddress, clock, noPings)

  case class Start(target: ActorSelection)

  sealed trait PingResult

  case object Up extends PingResult

  case object Unavailable extends PingResult

}

class Pinger(ctx: ExecutionContext, timeout: FiniteDuration, satelliteAddress: SatelliteAddress, clock: Clock, noPings: Int) extends Actor {

  def receive = waitingToStart

  def waitingToStart: Receive = {
    case Pinger.Start(target) =>
      if (!CommonSettings(context.system).satellite.enabled) {
        ctx.logError("Satellite has not been enabled on the XL Deploy server. Please ensure 'satellite.enabled = true' is set in the system.conf file of XL Deploy")
        sender() ! Unavailable
        context.stop(self)
      } else {
        ping(target)

        ctx.logOutput(s"Waiting for connection to satellite at $satelliteAddress")
        context.setReceiveTimeout(timeout)

        continueWith(sender()) {
          loggingPing(target, clock.currentTimeMillis(), noPings, sender())
        }
      }
  }

  def loggingPing(target: ActorSelection, startTime: Long, remainingPingToReceive: Int, requester: ActorRef): Receive = {
    case PingReply(uptime) =>
      if (remainingPingToReceive == noPings) ctx.logOutput(s"Satellite is live (uptime: $uptime sec)")

      val currentTime = clock.currentTimeMillis()

      logPing(startTime)

      if (remainingPingToReceive > 1) {

        ping(target)

        continueWith(requester) {
          loggingPing(target, currentTime, remainingPingToReceive - 1, requester)
        }

      } else {

        requester ! Up
        context stop self
      }
  }

  private def logPing(startTime: Long) {
    ctx.logOutput(s"Ping: ${clock.currentTimeMillis() - startTime} ms")
  }

  private def continueWith(requester: ActorRef)(nextBehavior: Receive) {
    context become (nextBehavior orElse handleTimeout(requester))
  }

  private def handleTimeout(requester: ActorRef): Receive = {
    case ReceiveTimeout =>
      ctx.logError(s"The satellite at $satelliteAddress cannot be reached. Please check whether it is running.")
      requester ! Unavailable
      context stop self
  }

  private def ping(target: ActorSelection) {
    target ! Ping
  }
}
