package com.xebialabs.deployit.plugin.satellite

import akka.actor._
import akka.pattern.ask
import akka.util.Timeout
import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.deployit.plugin.satellite.Pinger.{Unavailable, Up}
import com.xebialabs.satellite.protocol.{Paths, Ping, PingReply}
import com.xebialabs.xlplatform.satellite.Satellite
import com.xebialabs.xlplatform.settings.CommonSettings

import scala.concurrent.duration._
import scala.concurrent.{Await, ExecutionContextExecutor}

object Pinger {

  val FIVE_PINGS = 5

  def isUp(satellite: Satellite, noPings: Int = FIVE_PINGS)
          (implicit ctx: ExecutionContext, satelliteCommunicatorSystem: ActorSystem): Boolean =
    isUp(SatelliteAddress(satellite), ActorLocator(satellite), defaultPingTimeout(satelliteCommunicatorSystem), noPings, Clock())

  def defaultPingTimeout(satelliteCommunicatorSystem: ActorSystem): FiniteDuration =
    CommonSettings(satelliteCommunicatorSystem).satellite.pingTimeout.duration

  def isUp(satelliteAddress: SatelliteAddress, actorLocator: ActorLocator, pingTimeout: FiniteDuration,
           pingCount: Int, clock: Clock)
          (implicit ctx: ExecutionContext, satelliteCommunicatorSystem: ActorSystem): Boolean =
  {
    val grace = 1.second
    implicit val timeout: Timeout = Timeout(pingTimeout + grace) // allow Pinger to send back a result even after timeout
    implicit val dispatcher: ExecutionContextExecutor = satelliteCommunicatorSystem.dispatcher

    ctx.logOutput(s"Connecting to satellite at $satelliteAddress")
    val pinger = satelliteCommunicatorSystem.actorOf(Pinger.props(ctx, pingTimeout, satelliteAddress, clock, pingCount))
    val pingFuture = (pinger ? Pinger.Start(actorLocator.locate(Paths.ping))).mapTo[PingResult]
      .recover { case error =>
        ctx.logError(s"Operation failed  (${error.getMessage})")
        Unavailable
      }
    Await.result(pingFuture, timeout.duration + grace) == Up // more grace time allow local handling even after timeout
  }

  def props(ctx: ExecutionContext, timeout: FiniteDuration, satelliteAddress: SatelliteAddress, clock: Clock = Clock(),
            noPings: Int = Pinger.FIVE_PINGS) =
    Props(new Pinger(ctx, timeout, satelliteAddress, clock, noPings))

  sealed trait PingResult

  case class Start(target: ActorSelection)

  case object Up extends PingResult

  case object Unavailable extends PingResult

}

class Pinger(ctx: ExecutionContext, pingTimeout: FiniteDuration, satelliteAddress: SatelliteAddress, clock: Clock, pingCount: Int) extends Actor {

  def receive: Receive = if (!CommonSettings(context.system).satellite.enabled) notEnabled else waitingToStart

  def notEnabled: Receive = {
    case Pinger.Start(_) =>
      ctx.logError("Satellite has not been enabled on the XL Deploy server. " +
        "Please ensure 'xl.satellite.enabled = true' is set in the xl-deploy.conf file of XL Deploy")
      sender() ! Unavailable
      context.stop(self)
  }

  def waitingToStart: Receive = {
    case Pinger.Start(target) =>
      target ! Ping

      ctx.logOutput(s"Waiting for connection to satellite at $satelliteAddress")
      context.setReceiveTimeout(pingTimeout)

      context become awaitPings(target, clock.currentTimeMillis(), pingCount, sender())
  }

  def awaitPings(target: ActorSelection, startTime: Long, remainingPingsToReceive: Int, origRequester: ActorRef): Receive = {
    case PingReply(uptime) =>
      if (remainingPingsToReceive == pingCount) {
        ctx.logOutput(s"Satellite is live (uptime: $uptime sec)")
      }
      ctx.logOutput(s"Ping: ${clock.currentTimeMillis() - startTime} ms")

      if (remainingPingsToReceive > 1) {
        target ! Ping
        context become awaitPings(target, clock.currentTimeMillis(), remainingPingsToReceive - 1, origRequester)
      } else {
        origRequester ! Up
        context stop self
      }
    case ReceiveTimeout =>
      ctx.logError(s"The satellite at $satelliteAddress cannot be reached. Please check whether it is running.")
      origRequester ! Unavailable
      context stop self
  }
}
