/**
  * Copyright 2014-2018 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
  */
package com.xebialabs.deployit.plugin.satellite

import akka.actor.{Actor, ActorRef, ActorSelection, Props, ReceiveTimeout}
import com.xebialabs.deployit.engine.api.dto.SatelliteInfo
import com.xebialabs.deployit.plugin.satellite.SatelliteInfoChecker.Get
import com.xebialabs.satellite.protocol.{GetSatelliteInfo, SatelliteInfoReply}
import com.xebialabs.xlplatform.settings.CommonSettings
import grizzled.slf4j.Logging

object SatelliteInfoChecker {
  def props(satelliteAddress: String) = Props(new SatelliteInfoChecker(satelliteAddress))

  case class Get(target: ActorSelection)

}

class SatelliteInfoChecker(satelliteAddress: String) extends Actor with Logging {
  def receive = waitingForInfo

  private def waitingForInfo: Receive = {
    case Get(target) =>
      if (!CommonSettings(context.system).satellite.enabled) {
        logger.debug(s"Satellite $satelliteAddress is not enabled")
        sender() ! new SatelliteInfo()
        context stop self
      } else {
        target ! GetSatelliteInfo
        context become (waitingForReply(sender) orElse handleTimeout(sender))
      }
  }

  private def waitingForReply(origin: ActorRef): Receive = {
    case SatelliteInfoReply(info) =>
      origin ! info
      context become waitingForInfo
      context stop self
  }

  private def handleTimeout(origin: ActorRef): Receive = {
    case ReceiveTimeout =>
      logger.debug(s"The satellite $satelliteAddress cannot be reached. Please check whether it is running.")
      origin ! GetSatelliteInfo
      context stop self
  }

}
