package com.xebialabs.deployit.plugin.satellite

import java.io.BufferedInputStream
import java.net.InetSocketAddress

import akka.actor._
import akka.pattern.pipe
import akka.stream.scaladsl._
import akka.stream.{ActorMaterializer, ActorMaterializerSettings, ClosedShape}
import com.xebialabs.deployit.engine.tasker.satellite.Paths.tasks
import com.xebialabs.deployit.plugin.satellite.UploadTaskSupervisor.Protocol.{ErrorWithAttemptCount, FileToUpload, UploadConfig}
import com.xebialabs.satellite.protocol.UploadReply._
import com.xebialabs.satellite.protocol._
import com.xebialabs.satellite.streaming.DigesterStage.Digest
import com.xebialabs.satellite.streaming.SslStreamingSupport.SslConfig
import com.xebialabs.satellite.streaming._
import com.xebialabs.xlplatform.settings.CommonSettings

import scala.concurrent.{ExecutionContextExecutor, Future, Promise}
import scala.concurrent.duration.{Duration, FiniteDuration}
import scala.util.{Failure, Success, Try}

object FileUploader {
  def props(requester: ActorRef, config: UploadConfig, file: FileToUpload, attemptCount: Int): Props =
    Props(new FileUploader(requester, config, file, attemptCount))
}

class FileUploader(requester: ActorRef, config: UploadConfig, fileToUpload: FileToUpload, uploadAttempt: Int = 1)
  extends Actor with ActorLogging
{
  implicit val system: ActorSystem = context.system
  implicit val materializer: ActorMaterializer =
    ActorMaterializer(ActorMaterializerSettings(system).withDispatcher("streaming.StreamingDispatcher"))
  private val securitySettings = CommonSettings(system).security
  private val satelliteSettings = CommonSettings(system).satellite
  private[this] val startTime = System.currentTimeMillis()

  var uploadIdleTimeout: FiniteDuration = _

  override def preStart() {
    log.debug("FileUploader PreStart")
    val remoteUploadActor = config.satellite.locate(tasks)
    log.debug("Sending UploadFileForTask")
    remoteUploadActor ! UploadFileForTask(fileToUpload.id, config.taskId, fileToUpload.file.getName, fileToUpload.path)
  }

  def receive: Receive = reachingSatellite

  def reachingSatellite: Receive = {
    case InitConnection(port, chunkSize, compression, wantTls) =>
      val fileReceiver = sender()

      implicit val streamingConfig: StreamConfig = StreamConfig(compression, chunkSize)
      val satelliteReady = Promise[Unit]()

      Try {
        SslStreamingSupport.SslConfig(wantTls, securitySettings)
      }.flatMap(sslConfig => establishConnection(port, sslConfig, satelliteReady)) match {
        case Success(futureConnection) =>
          implicit val ec: ExecutionContextExecutor = context.dispatcher
          futureConnection.pipeTo(self)
          context.setReceiveTimeout(satelliteSettings.streamingConnectionTimeout.duration)
          context become waitingForConnection(fileReceiver, satelliteReady)
        case Failure(exc) =>
          handleConnectionError(s"Cannot enable SSL connection: ${exc.getMessage}", fileReceiver, Some(exc))
      }
    case AlreadyRegistered =>
      requester ! Error("Task is already registered")
      context stop self
  }

  private def establishConnection(port: Int, sslConfig: SslConfig, satelliteReady: Promise[Unit])
                                 (implicit streamingConfig: StreamConfig): Try[Future[Tcp.OutgoingConnection]] = {
    Try {
      val satelliteAddress = new InetSocketAddress(config.satelliteAddress.hostname, port)
      log.info(s"'connecting to $satelliteAddress' for upload '${fileToUpload.id}'")

      val fileStream = fileToUpload.file.getRawStream
      log.info("extracted stream from file to upload")
      val buffStream = new BufferedInputStream(fileStream)
      log.info("Sending file stream to  stream from file to upload")
      val byteSource = UploadStage.source(satelliteReady.future, buffStream) {
        case Digest(checksum, fileSize) => self ! StreamDone(checksum, fileSize)
      }

      val connection = Tcp().outgoingConnection(satelliteAddress)
      val wrappedConnection = SslStreamingSupport.wrapWithSsl(sslConfig.asClient.eagerClose, connection)
      val connectToDevNull =
        RunnableGraph.fromGraph(GraphDSL.create(byteSource, wrappedConnection)((_, c) => c) { implicit builder =>
          (bs, wrappedConn) =>
            import GraphDSL.Implicits._
            bs ~> wrappedConn ~> Sink.ignore
            ClosedShape.getInstance
        })
      connectToDevNull.run()
    }

  }

  def waitingForConnection(fileReceiver: ActorRef, satelliteReady: Promise[Unit]): Receive = {
    case connection@Tcp.OutgoingConnection(remoteAddress, localAddress) =>
      context.setReceiveTimeout(Duration.Undefined)
      log.info(s"Connected $localAddress -> $remoteAddress")
      fileReceiver ! Connected(connection.localAddress)
      log.info("Starting Sync operation...")
      context become inSync(fileReceiver, satelliteReady)
    case ReceiveTimeout =>
      handleConnectionError(
        s"Unable to establish connection within ${satelliteSettings.streamingConnectionTimeout.duration.toSeconds} seconds.",
        fileReceiver)
  }

  private def handleConnectionError(description: String, fileReceiver: ActorRef, exception: Option[Throwable] = None) {
    log.error(description)
    exception.foreach(e => log.debug(e.toString))
    fileReceiver ! CannotConnect(description)
    requester ! Error(description)
    context stop self
  }

  def inSync(fileReceiver: ActorRef, sync: Promise[Unit]): Receive = {
    case Ready =>
      sync.success(Unit)
      log.info("Sync operation successful, sending file to FileReceiver")
      context become (streaming(fileReceiver) orElse handleError)
  }

  def streaming(fileReceiver: ActorRef): Receive = {
    case StreamDone(checksum, fileLength) =>
      log.info(s"Sending $fileLength bytes to upload ")
      fileReceiver ! FileUploaded(checksum, fileLength)
      config.ctx.logOutput(
        s"""${fileToUpload.path}/${fileToUpload.file.getName} uploaded
           |- Size: $fileLength bytes
           |- Checksum: $checksum
           |- Transfertime: ${System.currentTimeMillis() - startTime}ms""".stripMargin)

    case Done =>
      log.info(s"'${fileToUpload.path}' uploaded successfully")
      requester ! Done
      context stop self
  }

  def handleError: Receive = {
    case Error(cause) =>
      log.warning(s"'${fileToUpload.file.getName}' failed")
      requester forward ErrorWithAttemptCount(cause, fileToUpload, uploadAttempt)
      context stop self
  }

  case class StreamDone(checksum: String, fileLength: Long)

}