package com.xebialabs.xlplatform.cluster.full.downing

import akka.actor.{ActorSystem, Address, Props}
import akka.cluster.DowningProvider
import com.typesafe.config.Config
import com.xebialabs.xlplatform.cluster.NodeState
import com.xebialabs.xlplatform.cluster.full.downing.LeaderAutoDowningActor.{DownAction, DownReachable, DownUnreachable}

import scala.concurrent.duration.{FiniteDuration, _}
import scala.language.postfixOps

class OldestLeaderAutoDowningProvider(system: ActorSystem) extends DowningProvider {

  val conf: Config = system.settings.config

  override def downRemovalMargin: FiniteDuration = conf.getDuration("akka.cluster.custom-downing.down-removal-margin").toMillis millis

  private val stableAfter = conf.getDuration("akka.cluster.custom-downing.stable-after").toMillis millis

  override def downingActorProps: Option[Props] = Some(OldestLeaderAutoDowning.props(stableAfter, downRemovalMargin))
}

object OldestLeaderAutoDowning {
  def props(stableAfter: FiniteDuration, downRemovalMargin: FiniteDuration): Props =
    Props(classOf[OldestLeaderAutoDowning], stableAfter, downRemovalMargin)
}

class OldestLeaderAutoDowning(stableAfter: FiniteDuration, downRemovalMargin: FiniteDuration) extends LeaderAutoDowningActor(stableAfter, downRemovalMargin) {

  def decide(): DownAction = {
    val unreachableSize = unreachableMembers.size
    val membersSize = members.size
    log.info(s"Deciding on current cluster state: $unreachableSize of $membersSize members unreachable.")

    val activeMembers = getActiveMembers

    members.find(member => activeMembers.contains(member.address)) match {
      case None =>
        log.info(s"Downing my partition [$reachable] - no active cluster members found.")
        DownReachable
      case Some(oldestMember) =>
        log.info(s"Current active oldest member is [${oldestMember.address}].")
        if (unreachable(oldestMember.address)) {
          log.info(s"Downing my partition [$reachable] - other partition has active oldest.")
          DownReachable
        } else {
          log.info(s"Downing other partition [$unreachable] - my partition has the current active oldest.")
          DownUnreachable
        }
    }
  }

  override def down(node: Address): Unit = {
    log.info("Downing member [{}].", node)
    cluster.down(node)
  }

  override def downSelf(): Unit = {
    log.info("Deactivating myself.")
    NodeState.setActive(false)
    AutoDowning.setDowning(true)
    log.info("Downing myself.")
    shutdownMember()
  }
}
