package com.xebialabs.xlplatform.cluster.full

import akka.actor.{LoggingFSM, Props}
import akka.cluster.Member
import com.xebialabs.xlplatform.cluster.NodeState
import com.xebialabs.xlplatform.cluster.membership.ClusterStateMessages.{ClusterMemberAdded, ClusterMemberLeft, Initialize, SelfAddedToCluster}

import scala.collection.immutable.SortedSet


object FullClusterNodeStateManager {
  def stateManager(initialize: () => Unit) = Props(classOf[FullClusterNodeStateManager], initialize)

  sealed trait State

  case object Initial extends State

  case object AwaitSelfRegistration extends State

  case object Standby extends State

  case object Active extends State

  sealed trait Data

  case class ClusterMembers(members: SortedSet[Member]) extends Data

  case class Brain(me: Member, members: SortedSet[Member]) extends Data

}

/**
  * Tracks state of the XLR specific "full cluster" akka node.
  *
  * @param doActivation
  */
class FullClusterNodeStateManager(doActivation: () => Unit)
  extends LoggingFSM[FullClusterNodeStateManager.State, FullClusterNodeStateManager.Data] {

  import FullClusterNodeStateManager._

  private implicit val executor = context.system.dispatcher

  when(Initial) {
    case Event(Initialize, _) =>
      stateLog("Initializing Cluster State...")
      goto(AwaitSelfRegistration) using ClusterMembers(collection.immutable.SortedSet.empty(Member.ageOrdering))
  }

  when(AwaitSelfRegistration) {
    case Event(ClusterMemberAdded(member), ClusterMembers(members)) =>
      stateLog(s"Cluster member $member joined")
      stay() using ClusterMembers(members + member)
    case Event(ClusterMemberLeft(member), ClusterMembers(members)) =>
      stateLog(s"Cluster member $member left")
      stay() using ClusterMembers(members - member)
    case Event(SelfAddedToCluster(me), ClusterMembers(members)) if members.isEmpty =>
      stateLog(s"I've joined the cluster and am the only one. Becoming active!")
      goto(Active) using Brain(me, members + me)
    case Event(SelfAddedToCluster(me), ClusterMembers(members)) =>
      stateLog(s"I've joined an active cluster, going to Active mode")
      goto(Active) using Brain(me, members + me)
  }

  onTransition {
    case _ -> Active =>
      activate()
  }

  when(Active) {
    case Event(ClusterMemberAdded(member), Brain(me, members)) =>
      stateLog(s"Cluster member $member joined")
      stay() using Brain(me, members + member)
    case Event(ClusterMemberLeft(member), Brain(me, members)) =>
      stateLog(s"Cluster member $member left")
      stay() using Brain(me, members - member)
  }

  startWith(Initial, ClusterMembers(SortedSet()))
  initialize()

  def activate(): Unit = {
    stateLog(s"Activating myself")
    NodeState.setActive(true)
    stateLog(s"Invoking activation callback")
    doActivation()
    stateLog(s"Fully activated!")
  }

  def stateLog(m: => String): Unit = {
    log.info(s"$stateName -- $m")
  }

}
