package com.xebialabs.xlplatform.cluster.membership

import akka.actor.{Actor, ActorRef, Address, Props}
import akka.cluster.ClusterEvent._
import akka.cluster.MemberStatus.Up
import akka.cluster.{Cluster, Member}
import com.xebialabs.xlplatform.cluster.NodeState
import com.xebialabs.xlplatform.cluster.membership.ClusterStateMessages.{ClusterMemberAdded, ClusterMemberLeft, Initialize, SelfAddedToCluster}
import com.xebialabs.xlplatform.cluster.membership.storage.ClusterMembershipManagement
import com.xebialabs.xlplatform.cluster.membership.storage.ClusterMembershipManagement.{Data, Seed}
import grizzled.slf4j.Logging

import scala.concurrent.Await
import scala.concurrent.duration.Duration


object ClusterStateMessages {

  sealed trait ClusterStateEvent

  case class SelfAddedToCluster(member: Member) extends ClusterStateEvent

  case object Initialize extends ClusterStateEvent

  case class ClusterMemberAdded(member: Member) extends ClusterStateEvent

  case class ClusterMemberLeft(member: Member) extends ClusterStateEvent

}

object ClusterMembershipManagingActor {
  def props(clusterStateManagingActor: ActorRef, clusterMembershipDialect: ClusterMembershipManagement) =
    Props(new ClusterMembershipManagingActor(clusterStateManagingActor, clusterMembershipDialect))
}

/**
  * This actor listens to the cluster messages and updates the clusterStateManagingActor.
  * If this actor sees that this node has become Cluster Leader, it will also remove seed nodes from the member table in the
  * persistent store.
  *
  * @param clusterStateManagingActor The actor that decides when this node is to become active, and deals with node registration
 *                                  in the DB
  * @param membershipMgmt The cluster membership registration object, in the dialect that the DB speaks.
  */
class ClusterMembershipManagingActor(clusterStateManagingActor: ActorRef, membershipMgmt: ClusterMembershipManagement)
  extends Actor with Logging {

  import context._

  lazy val cluster: Cluster = Cluster(system)

  lazy val me: Address = cluster.selfUniqueAddress.address

  @scala.throws[Exception](classOf[Exception])
  override def preStart(): Unit = {
    super.preStart()
    cluster.subscribe(self, InitialStateAsSnapshot, classOf[MemberUp], classOf[MemberRemoved], classOf[LeaderChanged])
  }

  @scala.throws[Exception](classOf[Exception])
  override def postStop(): Unit = {
    cluster.unsubscribe(self)
    super.postStop()
  }

  def deregisterSeedNode(member: Member): Unit = {
    Await.result(membershipMgmt.deregisterSeed(member.uniqueAddress.address), Duration.Inf)
  }

  def registerSeedNode(member: Member): Unit = {
    val memberAddress = member.uniqueAddress.address
      Await.result({
        membershipMgmt.listActiveSeeds(cluster).map {
          case Data(seeds: List[Seed]) =>
            if (!seeds.collect(_.address).contains(memberAddress)) {
              logger.debug(s"Registering cluster member: $member ")
              membershipMgmt.registerSeed(memberAddress)
            }
          case _ =>
            logger.warn(s"Not able to fetch list of active cluster seed nodes from the DB")
        }
      }, Duration.Inf)
  }

  def membershipAdministration: Receive = {
    case MemberUp(member) if member.isMe =>
      logger.debug(s"Admin: New cluster member: $member (that's me!)")
      clusterStateManagingActor ! SelfAddedToCluster(member)
      registerSeedNode(member)
    case MemberUp(member) =>
      logger.debug(s"Admin: New cluster member: $member")
      clusterStateManagingActor ! ClusterMemberAdded(member)
      registerSeedNode(member)
    case MemberRemoved(member, previousStatus) =>
      clusterStateManagingActor ! ClusterMemberLeft(member)
      logger.info(s"Member $member with status $previousStatus got removed from the cluster - deregistering it from the DB")
      deregisterSeedNode(member)

    case LeaderChanged(Some(leader)) if leader != cluster.selfAddress =>
      logger.debug(s"Cluster leader changed from me to $leader. I'll stop managing the cluster members DB. ($myNodeState)")
      context.become(membershipListening)
    case LeaderChanged(Some(leader)) if leader == cluster.selfAddress =>
      logger.debug(s"I already was the cluster leader. ($myNodeState)")
    case LeaderChanged(None) =>
      logger.warn(s"Cluster leader changed from me to NONE. I'll stop managing the cluster members DB. ($myNodeState)")
      context.become(membershipListening)
  }

  def membershipListening: Receive = {
    case MemberUp(member) if member.isMe =>
      logger.debug(s"Listening: New cluster member: $member (that's me!)")
      clusterStateManagingActor ! SelfAddedToCluster(member)
    case MemberUp(member) =>
      logger.debug(s"Listening: New cluster member: $member")
      clusterStateManagingActor ! ClusterMemberAdded(member)
    case MemberRemoved(member, previousStatus) =>
      logger.info(s"Member $member with status $previousStatus got removed from the cluster")
      clusterStateManagingActor ! ClusterMemberLeft(member)

    case LeaderChanged(Some(leader)) if leader == cluster.selfAddress =>
      logger.info(s"I have become the cluster leader and will now manage the cluster members DB. ($myNodeState)")
      context.become(membershipAdministration)
    case LeaderChanged(Some(leader)) if leader != cluster.selfAddress =>
      logger.debug(s"The cluster leader has changed to $leader. ($myNodeState)")
    case LeaderChanged(None) =>
      logger.debug(s"The cluster leader has changed to NONE. ($myNodeState)")
  }

  override def receive: Receive = {
    case CurrentClusterState(members, unreachableMembers, _, optLeader, _) =>
      logger.info(s"Received cluster state with $members")
      if (unreachableMembers.nonEmpty) {
        logger.info(s"These members are currently unreachable: $unreachableMembers")
      }
      val iAmTheLeader = optLeader.contains(cluster.selfAddress)
      if (optLeader.isEmpty) {
        logger.warn(s"The cluster ${context.system.name} currently has no leader")
      } else {
        logger.info(s"The current cluster leader is ${optLeader.get} ${if (iAmTheLeader) "(me!)" else ""}")
      }
      clusterStateManagingActor ! Initialize
      members.filter(_.status == Up).foreach { member =>
        if (member.isMe) {
          clusterStateManagingActor ! SelfAddedToCluster(member)
        } else {
          clusterStateManagingActor ! ClusterMemberAdded(member)
        }
      }
      logger.debug(myNodeState)
      val nextState = if (iAmTheLeader) membershipAdministration else membershipListening
      context.become(nextState)
  }


  private implicit class ExtendedMember(member: Member) {
    def isMe: Boolean = {
      member.uniqueAddress == cluster.selfUniqueAddress
    }
  }

  private def myNodeState: String = s"I am in state [${if (NodeState.isActive) "Active" else "Stand-by"}]"

}
