package com.xebialabs.xlplatform.cluster.hotstandby

import akka.actor.{LoggingFSM, Props}
import akka.cluster.Member
import com.xebialabs.xlplatform.cluster.NodeState
import com.xebialabs.xlplatform.cluster.membership.ClusterStateMessages.{ClusterMemberAdded, ClusterMemberLeft, Initialize, SelfAddedToCluster}

import scala.collection.immutable.SortedSet
import scala.concurrent.ExecutionContextExecutor

object HotStandbyClusterNodeStateManager {
  def stateManager(initialize: () => Unit) = Props(new HotStandbyClusterNodeStateManager(initialize))

  sealed trait State

  case object Initial extends State

  case object AwaitSelfRegistration extends State

  case object Standby extends State

  case object Active extends State

  sealed trait Data

  case class ClusterMembers(members: SortedSet[Member]) extends Data

  case class Brain(me: Member, members: SortedSet[Member]) extends Data

}

class HotStandbyClusterNodeStateManager(doActivation: () => Unit)
  extends LoggingFSM[HotStandbyClusterNodeStateManager.State, HotStandbyClusterNodeStateManager.Data] {

  import HotStandbyClusterNodeStateManager._

  private implicit val executor: ExecutionContextExecutor = context.system.dispatcher

  def stateLog(m: => String): Unit = {
    log.info(s"$stateName -- $m")
  }

  when(Initial) {
    case Event(Initialize, _) =>
      stateLog("Initializing Cluster State...")
      goto(AwaitSelfRegistration) using ClusterMembers(collection.immutable.SortedSet.empty(Member.ageOrdering))
  }

  when(AwaitSelfRegistration) {
    case Event(ClusterMemberAdded(member), ClusterMembers(members)) =>
      stateLog(s"Cluster member $member joined")
      stay() using ClusterMembers(members + member)
    case Event(ClusterMemberLeft(member), ClusterMembers(members)) =>
      stateLog(s"Cluster member $member left")
      stay() using ClusterMembers(members - member)
    case Event(SelfAddedToCluster(me), ClusterMembers(members)) if members.isEmpty =>
      stateLog(s"I've joined the cluster and am the only one. Becoming active!")
      goto(Active) using Brain(me, members + me)
    case Event(SelfAddedToCluster(me), ClusterMembers(members)) =>
      stateLog(s"I've joined an active cluster, going to Standby mode")
      goto(Standby) using Brain(me, members + me)
  }

  def activate(): Unit = {
    stateLog(s"Taking over the cluster")
    NodeState.setActive(true)
    stateLog(s"Invoking activation callback")
    doActivation()
    stateLog(s"Fully activated!")
  }

  onTransition {
    case _ -> Active =>
      activate()
  }

  when(Active) {
    case Event(ClusterMemberAdded(member), Brain(me, members)) =>
      stateLog(s"Cluster member $member joined")
      stay() using Brain(me, members + member)
    case Event(ClusterMemberLeft(member), Brain(me, members)) =>
      stateLog(s"Cluster member $member left")
      stay() using Brain(me, members - member)
  }

  when(Standby) {
    case Event(ClusterMemberAdded(member), Brain(me, members)) =>
      stateLog(s"Cluster member $member joined")
      stay() using Brain(me, members + member)
    case Event(ClusterMemberLeft(member), Brain(me, members)) =>
      stateLog(s"Cluster member $member left")
      val newMembers: SortedSet[Member] = members - member
      if (newMembers.headOption.exists(_.uniqueAddress == me.uniqueAddress)) {
        stateLog("I'm the oldest one, become Active")
        goto(Active) using Brain(me, newMembers)
      } else {
        stay() using Brain(me, newMembers)
      }
  }

  startWith(Initial, ClusterMembers(SortedSet()))
  initialize()

}

