package com.xebialabs.xlplatform.cluster.full.downing

import com.typesafe.config.Config
import com.xebialabs.xlplatform.cluster.NodeState
import com.xebialabs.xlplatform.cluster.full.downing.LeaderAutoDowningActor.{DownAction, DownReachable, DownUnreachable}
import org.apache.pekko.actor.{ActorSystem, Address, Props}
import org.apache.pekko.cluster.DowningProvider

import scala.concurrent.duration.{FiniteDuration, _}
import scala.language.postfixOps

class MajorityLeaderAutoDowningProvider(system: ActorSystem) extends DowningProvider {

  val conf: Config = system.settings.config

  override def downRemovalMargin: FiniteDuration = conf.getDuration("pekko.cluster.custom-downing.down-removal-margin").toMillis millis

  private val stableAfter = conf.getDuration("pekko.cluster.custom-downing.stable-after").toMillis millis

  override def downingActorProps: Option[Props] = Some(MajorityLeaderAutoDowning.props(stableAfter, downRemovalMargin))
}

object MajorityLeaderAutoDowning {
  def props(stableAfter: FiniteDuration, downRemovalMargin: FiniteDuration): Props =
    Props(classOf[MajorityLeaderAutoDowning], stableAfter, downRemovalMargin)
}

class MajorityLeaderAutoDowning(stableAfter: FiniteDuration, downRemovalMargin: FiniteDuration) extends LeaderAutoDowningActor(stableAfter, downRemovalMargin) {

  def decide(): DownAction = {
    val unreachableSize = unreachableMembers.size
    val membersSize = members.size
    log.info(s"Deciding on current cluster state: $unreachableSize of $membersSize members unreachable.")

    if (unreachableSize * 2 == membersSize) {
      log.info("Both partitions are equal in size, break the tie by keeping the side with oldest member.")
      val oldestMember = members.head.address
      val oldestIsActive = isActiveMember(oldestMember)
      log.info(s"Oldest member [$oldestMember] detected as being ${if (oldestIsActive) "active" else "inactive"}.")
      if (unreachable(oldestMember) && oldestIsActive) {
        log.info(s"Downing my partition [$reachable] - other partition has active oldest.")
        DownReachable
      } else if (reachable(oldestMember) && !oldestIsActive) {
        // edge case when oldest does not have connectivity to DB
        log.info(s"Downing my partition [$reachable] - my partition has invalid oldest.")
        DownReachable
      } else {
        log.info(s"Downing other partition [$unreachable] - my partition has the current active oldest.")
        DownUnreachable
      }
    } else if (unreachableSize * 2 < membersSize) {
      log.info(s"We are in majority, downing other partition [$unreachable].")
      DownUnreachable
    } else {
      log.info(s"We are in minority, downing my partition [$reachable].")
      DownReachable
    }
  }

  override def down(node: Address): Unit = {
    log.info("Downing member [{}].", node)
    cluster.down(node)
  }

  override def downSelf(): Unit = {
    log.info("Deactivating myself.")
    NodeState.setActive(false)
    AutoDowning.setDowning(true)
    log.info("Downing myself.")
    shutdownMember()
  }
}


