package com.xebialabs.xlplatform.cluster

import com.xebialabs.xlplatform.cluster.ClusterMode.{Full, HotStandby}
import com.xebialabs.xlplatform.cluster.full.FullClusterNodeStateManager
import com.xebialabs.xlplatform.cluster.hotstandby.HotStandbyClusterNodeStateManager
import com.xebialabs.xlplatform.cluster.membership.ClusterDiscoveryActor
import com.xebialabs.xlplatform.cluster.membership.storage.{ClusterMembershipManagement, ClusterMembershipSQLManagement}
import com.zaxxer.hikari.HikariDataSource
import org.apache.pekko.actor.{ActorSystem, ExtendedActorSystem, Extension, ExtensionId, ExtensionIdProvider}
import org.apache.pekko.cluster.Cluster

import java.util.concurrent.atomic.AtomicReference

object XlCluster extends ExtensionId[XlClusterProvider] with ExtensionIdProvider {
  var config: ClusterConfig = _
  private var xlClusterProviderHolder: Option[XlClusterProvider] = None

  def getXlClusterProvider: Option[XlClusterProvider] = xlClusterProviderHolder

  val cluster = new AtomicReference[Cluster]
  val membershipManagement = new AtomicReference[ClusterMembershipSQLManagement]
  val actorSystem = new AtomicReference[ActorSystem]

  def getCluster: Cluster = cluster.get()
  def getMembershipManagement: ClusterMembershipSQLManagement = membershipManagement.get()
  def getActorSystem: ActorSystem = actorSystem.get()

  def init(config: ClusterConfig): Unit = {
    this.config = config
  }

  override def lookup(): ExtensionId[_ <: Extension] = this

  override def createExtension(system: ExtendedActorSystem): XlClusterProvider = new XlClusterProvider(system, config)

  override def get(system: ActorSystem): XlClusterProvider = {
    xlClusterProviderHolder = Some(super.get(system))
    xlClusterProviderHolder.get
  }
}


class XlClusterProvider(extendedSystem: ExtendedActorSystem, config: ClusterConfig) extends Extension {
  private implicit val system = extendedSystem

  private implicit val executionContext = extendedSystem.dispatcher

  private val cluster = Cluster(system)

  val membershipManagement = ClusterMembershipManagement.selectDialect(config.membership.datasource, config.membership.ttl)

  def doWithDatasource(callback: HikariDataSource => Unit): Unit = {
    callback(membershipManagement.datasource)
  }

  def start(initialize: () => Unit): Unit = {
    val clusterStateManagerActor = config.mode match {
      case Full =>
        system.actorOf(FullClusterNodeStateManager.stateManager(initialize))
      case HotStandby =>
        system.actorOf(HotStandbyClusterNodeStateManager.stateManager(initialize))
      case _ => throw new IllegalArgumentException("This extension is only valid with cluster usage.")
    }
    val discoveryActor = system.actorOf(ClusterDiscoveryActor.props(cluster, membershipManagement, config.membership.heartbeatInterval))
    discoveryActor ! ClusterDiscoveryActor.Start(clusterStateManagerActor)
    XlCluster.cluster.set(cluster)
    XlCluster.membershipManagement.set(membershipManagement)
    XlCluster.actorSystem.set(system)
  }

}
