package com.xebialabs.license;


import java.io.*;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.context.support.SpringBeanAutowiringSupport;
import com.samskivert.mustache.Mustache;
import com.samskivert.mustache.Template;

import com.xebialabs.license.service.LicenseInstallService;
import com.xebialabs.license.service.LicenseService;

import static com.xebialabs.license.License.PRODUCT_XL_DEPLOY;
import static com.xebialabs.license.License.PRODUCT_XL_RELEASE;
import static java.lang.String.format;

public class LicenseRegistrationServlet extends HttpServlet {

    public static final String PRODUCT_NAME = "productName";
    public static final String PRODUCT_LOGO_URL = "productLogoUrl";

    public static final String REGISTRATION_TEMPLATE_FILENAME = "registration.mustache";
    public static final String REGISTRATION_VIEW_MODEL_KEY = "registration-view-model";
    public static final String FONT_LOCATION_EOT = "fontLocationEot";
    public static final String FONT_LOCATION_TTF = "fontLocationTtf";
    public static final String FAVICON = "favicon";
    public static final String FAVICON_TYPE = "faviconType";
    public static final String ACTIVATION_URL = "activationUrl";

    private Logger logger = LoggerFactory.getLogger(LicenseRegistrationServlet.class);

    @Autowired
    private LicenseInstallService licenseInstallService;

    @Autowired
    private LicenseService licenseService;

    private String productName = "{{Define productName init-param}}";
    private String productLogoUrl = "";
    private String fontLocationEot = "fonts/open-sans/OpenSans-Regular-webfont.eot";
    private String fontLocationTtf = "fonts/open-sans/OpenSans-Regular-webfont.ttf";
    private String favicon = "favicon.ico";
    private String faviconType = "image/x-icon";
    private String activationUrl = "";

    public LicenseRegistrationServlet(LicenseInstallService licenseInstallService, LicenseService licenseService) {
        this.licenseInstallService = licenseInstallService;
        this.licenseService = licenseService;
    }

    // required so that Servlet can be instantiated
    public LicenseRegistrationServlet() {
    }

    @Override
    public void init(ServletConfig config) throws ServletException {
        super.init(config);
        SpringBeanAutowiringSupport.processInjectionBasedOnCurrentContext(this);

        if (config.getInitParameter(PRODUCT_NAME) != null) {
            productName = config.getInitParameter(PRODUCT_NAME);
        }
        if (config.getInitParameter(PRODUCT_LOGO_URL) != null) {
            productLogoUrl = config.getInitParameter(PRODUCT_LOGO_URL);
        }
        if (config.getInitParameter(FONT_LOCATION_EOT) != null) {
            fontLocationEot = config.getInitParameter(FONT_LOCATION_EOT);
        }
        if (config.getInitParameter(FONT_LOCATION_TTF) != null) {
            fontLocationTtf = config.getInitParameter(FONT_LOCATION_TTF);
        }
        if (config.getInitParameter(FAVICON) != null) {
            favicon = config.getInitParameter(FAVICON);
        }
        if (config.getInitParameter(FAVICON_TYPE) != null) {
            faviconType = config.getInitParameter(FAVICON_TYPE);
        }
        if (config.getInitParameter(ACTIVATION_URL) != null) {
            activationUrl = config.getInitParameter(ACTIVATION_URL);
        }
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        renderTemplate(resp, new RegistrationViewModel(req.getContextPath()), REGISTRATION_TEMPLATE_FILENAME);
    }

    @Override
    protected void doPost(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        RegistrationViewModel registrationViewModel = new RegistrationViewModel(req.getContextPath());

        String licenseText = req.getParameter("license");

        if (licenseTextIsProvided(licenseText)) {
            registrationViewModel.setLicenseText(licenseText);

            License license;
            try {
                license = licenseInstallService.installNewLicense(licenseText);
                if (license != null) {
                    registrationViewModel.setCanUseProduct();
                    LocalDate validUntil = license.getLocalDateValue(LicenseProperty.EXPIRES_AFTER);
                    if (validUntil != null) {
                        registrationViewModel.setValidMessage(format("Your license is valid until %s.", validUntil.toString("MMMM d, yyyy")));
                    } else {
                        registrationViewModel.setValidMessage("Your license is valid.");
                    }
                    registrationViewModel.setLicense(license);
                } else {
                    registrationViewModel.setWarningMessage("The license key provided is not valid.");
                }
            } catch (LicenseRepositoryIdException e) {
                registrationViewModel.setWarningMessage("This installation is locked to a previous license. Please renew your license.");
            } catch (LicenseViolationException e) {
                registrationViewModel.setWarningMessage(format("There is a problem with the provided license key: %s.", e.getMessage()));
            } catch (Exception e) {
                logger.warn("Failed to install license with license text: [{}]", licenseText, e);
                registrationViewModel.setWarningMessage(format("An error occurred while installing the license key: %s.", e.getMessage()));
            }
        } else {
            registrationViewModel.setWarningMessage("No license key was provided. Enter a license key in the field above.");
        }

        pageContent(registrationViewModel);
        renderTemplate(resp, registrationViewModel, REGISTRATION_TEMPLATE_FILENAME);
    }

    private void pageContent(RegistrationViewModel registrationViewModel) {
        // Do upgrade scenario
        registrationViewModel.setLicenseRenewalMessage(licenseInstallService.getLicenseRenewalMessage());
        registrationViewModel.setLicense(licenseService.getLicense());
    }

    private boolean licenseTextIsProvided(String licenseText) {
        return licenseText != null && !"".equals(licenseText);
    }

    private void renderTemplate(HttpServletResponse resp, RegistrationViewModel registrationViewModel, String templateFile) throws IOException {
        InputStream resource = this.getClass().getResourceAsStream(templateFile);
        InputStreamReader reader = new InputStreamReader(resource);
        Mustache.Compiler compiler = Mustache.compiler().emptyStringIsFalse(true).defaultValue("");
        Template templ = compiler.compile(reader);

        Writer writer = resp.getWriter();
        resp.setContentType("text/html");
        templ.execute(registrationViewModel.toScope(), writer);
    }

    class RegistrationViewModel {
        private final String contextPath;
        private String validMessage = "";
        private String warningMessage = "";
        private String submitLicenseState = ""; // enabled
        private String useProductState = "disabled";
        private String licenseText = "";
        private License license;
        private String licenseRenewalMessage;

        public RegistrationViewModel(String contextPath) {
            if (contextPath.endsWith("/")) {
                this.contextPath = contextPath;
            } else {
                this.contextPath = contextPath + "/";
            }
        }

        public void setValidMessage(String message) {
            this.warningMessage = "";
            this.validMessage = message;
        }

        public void setWarningMessage(String message) {
            this.warningMessage = message;
            this.validMessage = "";
        }

        public Map<String, Object> toScope() {
            Map<String, Object> scope = new HashMap<>();

            // customizable properties
            scope.put(PRODUCT_NAME, productName);
            scope.put(PRODUCT_LOGO_URL, productLogoUrl);
            scope.put(FONT_LOCATION_TTF, fontLocationTtf);
            scope.put(FONT_LOCATION_EOT, fontLocationEot);
            scope.put(FAVICON, favicon);
            scope.put(FAVICON_TYPE, faviconType);
            // end of customizable properties

            scope.put("contextPath", contextPath);
            scope.put("activationUrl", activationUrl);

            // validation message
            scope.put("valid", validMessage);
            scope.put("warning", warningMessage);

            // states of the two buttons
            scope.put("submitLicenseState", submitLicenseState);
            scope.put("useProductState", useProductState);

            scope.put("licenseText", licenseText);
            scope.put("licenseRenewalMessage", licenseRenewalMessage);

            // License data in modal
            if (license != null && !license.isDummyLicense()) {
                String productString = license.getStringValue(LicenseProperty.PRODUCT);

                scope.put("product", productString);
                scope.put("licensedTo", license.getStringValue(LicenseProperty.LICENSED_TO));
                String contact = license.getStringValue(LicenseProperty.CONTACT);
                scope.put("contact", contact);
                scope.put("repositoryId", license.getStringValue(LicenseProperty.REPOSITORY_ID));
                scope.put("expiresAfter", license.getStringValue(LicenseProperty.EXPIRES_AFTER));
                scope.put("contactEmail", encodedEmailFromContact(contact));

                if (license.getStringValue(LicenseProperty.EDITION).equals(LicenseVersion4.Edition4.Unregistered.name())) {
                    String productKey = "unknown";
                    if (productString.equals(PRODUCT_XL_DEPLOY)) {
                        productKey = "deploy";
                    } else if (productString.equals(PRODUCT_XL_RELEASE)) {
                        productKey = "release";
                    }
                    scope.put("productUnregisteredKey", productKey);
                }
            }
            // /License data in modal

            scope.put("copyrightYear", new LocalDate().getYear());
            return scope;
        }

        public void setCanUseProduct() {
            this.submitLicenseState = "disabled";
            this.useProductState = ""; // enabled
        }

        public void setLicenseText(String licenseText) {
            this.licenseText = licenseText;
        }

        public void setLicense(License license) {
            this.license = license;
        }

        public void setLicenseRenewalMessage(String licenseRenewalMessage) {
            this.licenseRenewalMessage = licenseRenewalMessage;
        }
    }

    static String encodedEmailFromContact(String contact) {
        String email;
        final Pattern compile = Pattern.compile(".*<.*@.*>");
        if (compile.matcher(contact).matches()) {
            email = contact.substring(contact.indexOf('<') + 1, contact.indexOf('>'));
        } else {
            email = contact;
        }
        try {
            return URLEncoder.encode(email, StandardCharsets.UTF_8.name());
        } catch (UnsupportedEncodingException e) {
            return contact;
        }
    }

}
