package com.xebialabs.license;

import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.license.service.LicenseService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.context.support.SpringBeanAutowiringSupport;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.HashSet;

import static javax.servlet.http.HttpServletResponse.SC_PAYMENT_REQUIRED;


public class LicenseValidationFilter implements Filter {

    private static final Logger logger = LoggerFactory.getLogger(LicenseValidationFilter.class);

    public static final String WHITE_LISTED_PATHS = "whiteListedPaths";

    @Autowired
    private LicenseService licenseService;

    private HashSet<String> whiteList = new HashSet<>();

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
        logger.trace("Added LicenseCheckFilter to the mix.");
        SpringBeanAutowiringSupport.processInjectionBasedOnCurrentContext(this);

        String whiteListedPathsString = filterConfig.getInitParameter(WHITE_LISTED_PATHS);
        whiteList = new HashSet<>(Strings.split(whiteListedPathsString, ","));
    }

    public void setLicenseService(LicenseService licenseService) {
        Checks.checkNotNull(licenseService, "LicenseCheckFilter: Could not find license check service");
        this.licenseService = licenseService;
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        if (!(request instanceof HttpServletRequest && response instanceof HttpServletResponse)) return;

        License license = licenseService.getLicense();

        // Note: Path info does not include the context root.
        String pathInfo = ((HttpServletRequest) request).getPathInfo();

        if (whiteList.contains(pathInfo)) {
            logger.debug("Path {} is white listed", pathInfo);
            chain.doFilter(request, response);
        } else if (license.isDummyLicense() || !licenseIsValid()) {
            logger.debug("This license is invalid: {}", license);
            ((HttpServletResponse) response).sendError(SC_PAYMENT_REQUIRED, "License expired");
        } else {
            chain.doFilter(request, response);
        }
    }

    private boolean licenseIsValid() {
        try {
            licenseService.validate();
        } catch (LicenseViolationException e) {
            logger.debug("Invalid license:", e);
            return false;
        }
        return true;
    }

    @Override
    public void destroy() {
        // Empty
    }
}
