package com.xebialabs.analytics.pendo

import com.xebialabs.analytics.pendo.PendoDataPublisher.{DisablePublishing, EnablePublishing, Tick}
import grizzled.slf4j.Logging
import org.apache.pekko.actor.{Actor, ActorSystem, Props, Scheduler}
import org.apache.pekko.http.scaladsl.Http
import org.apache.pekko.http.scaladsl.model._
import org.apache.pekko.http.scaladsl.model.headers.RawHeader
import org.apache.pekko.util.ByteString
import org.codehaus.jettison.json.JSONObject

import scala.concurrent.duration.{DurationInt, FiniteDuration}
import scala.concurrent.{ExecutionContextExecutor, Future}
import scala.util.{Failure, Success}

object PendoDataPublisher {

  def props(queue: PendoEventQueue, sender: PendoSender, scheduler: Scheduler): Props = Props(new PendoDataPublisher(queue, sender, scheduler))

  private[pendo] case object Tick

  case object EnablePublishing

  case object DisablePublishing
}

class PendoDataPublisher(queue: PendoEventQueue, sender: PendoSender, scheduler: Scheduler) extends Actor with Logging {

  private implicit val executionContext: ExecutionContextExecutor = context.system.dispatcher

  private def scheduleNextTickIn(duration: FiniteDuration) = {
    scheduler.scheduleOnce(duration, new Runnable {
      override def run(): Unit = self ! Tick
    })
  }

  override def receive: Receive = disabled

  def enabled: Receive = {
    case Tick =>
      logger.debug(s"Pendo task triggered, queue size = ${queue.size}")
      queue.peek() match {
        case Some(event) => sender.send(event).onComplete {
          case Success(_) =>
            queue.poll()
            scheduleNextTickIn(1.second)
          case Failure(e) => //TODO handle (i.e. skip) events that pendo doesn't and _will never_ want.
            scheduleNextTickIn(1.minute)
            logger.debug(s"$event not sent to Pendo", e)
        }
        case None => scheduleNextTickIn(1.second)
      }
    case DisablePublishing =>
      logger.debug(s"Clearing Pendo event queue")
      queue.clear()
      context.become(disabled)
  }

  def disabled: Receive = {
    case EnablePublishing =>
      context.become(enabled)
      scheduleNextTickIn(1.second)
    case DisablePublishing =>
      logger.debug("PendoDataPublisher is in a disabled state. Handling DisablePublishing message is a noop in a disabled state.")
  }
}

class PendoSender(pendoAccountHolder: PendoAccountHolder, trackedEventIntegrationKey: String)(implicit actorSystem: ActorSystem) extends Logging {

  def send(event: PendoEvent): Future[HttpResponse] = {
    Http().singleRequest(createPendoRequest(event))
  }

  private def createPendoRequest(event: PendoEvent) = {
    val entity = createEntity(event).toString(2)
    logger.debug(s"Sending $entity to Pendo")
    HttpRequest(method = HttpMethods.POST,
      uri = "https://app.pendo.io/data/track",
      headers = List(RawHeader("x-pendo-integration-key", trackedEventIntegrationKey)),
      HttpEntity.Strict(ContentType(MediaTypes.`application/json`), ByteString(entity.getBytes)))
  }

  private def createEntity(event: PendoEvent): JSONObject = {
    val entity = new JSONObject()
    val properties = new JSONObject()
    event.properties.foreach(entry => properties.put(entry._1, entry._2))
    entity.put("type", "track")
    entity.put("event", event.eventType)
    entity.put("visitorId", pendoAccountHolder.getVisitorId)
    entity.put("accountId", pendoAccountHolder.getAccountId)
    entity.put("timestamp", event.eventTime)
    entity.put("properties", properties)
    entity.put("context", new JSONObject())
    entity
  }
}
