package com.xebialabs.plugin.manager.cli

import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.plugin.manager.materializer.FilePlugin
import com.xebialabs.plugin.manager.sql.{DbPlugin, SqlPluginRepository}
import com.xebialabs.xlplatform.sugar.PathSugar.path2File
import org.apache.commons.io.FileUtils

import java.io.{BufferedReader, InputStreamReader}
import java.nio.file.{Files, Path, Paths}
import scala.jdk.CollectionConverters.IteratorHasAsScala

class PluginManagerCli(pluginRepository: SqlPluginRepository, extension: String) {

  def run(): Unit = {
    ConfigWrapper.extension = extension
    println("------Welcome to Plugin Manager CLI------")
    implicit var pluginsFromDb: Seq[DbPlugin] = pluginRepository.getAllWithoutBytes
    implicit var pluginsFromFs: Seq[FilePlugin] = getFromFilesystem()

    if (pluginsFromDb.nonEmpty || pluginsFromFs.nonEmpty) {
      var deleteAnotherPlugin: Boolean = true
      val stdin = new BufferedReader(new InputStreamReader(System.in))

      while ((pluginsFromDb.nonEmpty || pluginsFromFs.nonEmpty) && deleteAnotherPlugin) {

        printPlugins(pluginsFromDb, pluginsFromFs)

        askForPluginToDelete(stdin) {delete}

        println("Do you want to delete another plugin? yes/no")
        if (stdin.readLine().equalsIgnoreCase("yes")) {
          pluginsFromDb = pluginRepository.getAllWithoutBytes
          pluginsFromFs = getFromFilesystem()
        } else deleteAnotherPlugin = false
      }
    } else println("------No plugins in database or file system------")

    println("------Shutting down Plugin Manager CLI------")
    System.exit(0)
  }


  def getFromFilesystem(): Seq[FilePlugin] = {
    val pluginsDir: Path = Paths.get("plugins")
    Files.walk(pluginsDir)
      .filter(_.isFile)
      .filter(file => file.getFileName.toString.endsWith(ConfigWrapper.extension))
      .map(filePath => FilePlugin(filePath))
      .iterator().asScala.toSeq
  }

  def askForPluginToDelete(stdin: BufferedReader)(callback: String => Unit)(implicit dbPlugins: Seq[DbPlugin], fsPlugins: Seq[FilePlugin]): Unit = {
    println("If you want to exit Plugin Manager CLI type q or quit ")
    println("Which plugin do you want to delete? (Please type full name without version)")
    print("> ")
    val userInput = stdin.readLine

    if (userInput.equalsIgnoreCase("q") || userInput.equalsIgnoreCase("quit")) {
      println("------Shutting down Plugin Manager CLI------")
      System.exit(0)
    }
    else {
      if (dbPlugins.exists(dbPlugin => dbPlugin.name.equals(userInput)) || fsPlugins.exists(fsPlugin => fsPlugin.name.equals(userInput))) {
        callback.apply(userInput)
      } else {
        println("This is not a valid plugin name.")
      }
    }
  }

  def delete(userInput: String)(implicit fsPlugins: Seq[FilePlugin], dbPlugins: Seq[DbPlugin]): Unit = {
    if (fsPlugins.exists(fsPlugin => fsPlugin.name.equals(userInput))) {
      deleteFromFs(userInput)
    }
    if (dbPlugins.exists(dbPlugin => dbPlugin.name.equals(userInput))) {
      deleteFromDb(userInput)
    }
    println("Please verify and delete plugin file in other cluster members' plugins directory if needed")
  }

  def deleteFromFs(userInput: String)(implicit plugins: Seq[FilePlugin]): Unit = {
    FileUtils.forceDelete(plugins.filter(plugin => plugin.name == userInput).head.filePath)
    println(s"$userInput deleted from filesystem")
  }

  def deleteFromDb(userInput: String)(implicit plugins: Seq[DbPlugin]): Unit = {
    pluginRepository.delete(plugins.filter(plugin => plugin.name == userInput).head)
    println(s"$userInput deleted from database")
  }
  private def printPlugins(dbPlugins: Seq[DbPlugin], fsPlugins: Seq[FilePlugin]): Unit = {
    println(String.format("%-10s%-40s%-20s%-10s%-10s", "NUMBER", "NAME", "VERSION", "DATABASE", "FILESYSTEM"))
    println("-------------------------------------------------------------------------------------------")

    var pluginCounter = 1
    dbPlugins.foreach(plugin => {
      val existInFs = fsPlugins.exists(fsPlugin => fsPlugin.name.equals(plugin.name) && fsPlugin.version.equals(plugin.version.getOrElse("")))
      println(String.format("%-10s%-40s%-20s%-10s%-14s", pluginCounter, plugin.name, plugin.version.getOrElse(""), "+", if (existInFs) "+" else "-"))
      pluginCounter += 1
    })
    fsPlugins.foreach(plugin => {
      if (!dbPlugins.exists(fsPlugin => fsPlugin.name.equals(plugin.name) && fsPlugin.version.getOrElse("").equals(plugin.version))) {
        println(String.format("%-10s%-40s%-20s%-10s%-14s", pluginCounter, plugin.name, plugin.version, "-", "+"))
      }
      pluginCounter += 1
    })
    println("-------------------------------------------------------------------------------------------")
  }
}
