package com.xebialabs.plugin.manager.service

import com.xebialabs.deployit.server.api.upgrade.{Version => PlatformVersion}
import com.xebialabs.plugin.manager._
import com.xebialabs.plugin.manager.config.{ConfigWrapper, PluginManagerProperties}
import com.xebialabs.plugin.manager.metadata.{Version, XLProduct}
import com.xebialabs.plugin.manager.repository.PluginsRepository
import com.xebialabs.plugin.manager.repository.nexus.NexusRepositoryConfig.ListMethod
import com.xebialabs.plugin.manager.repository.nexus.{NexusPluginRepository, NexusRepositoryConfig}
import com.xebialabs.plugin.manager.repository.sql.SqlPluginRepository
import grizzled.slf4j.Logging

import scala.collection.mutable
import scala.util.{Failure, Success}


class LocalPluginService(config: PluginManagerProperties, product: XLProduct, version: Version, sqlPluginRepository: SqlPluginRepository) extends PluginService with Logging {

  def this(config: PluginManagerProperties, product: String, version: PlatformVersion, sqlPluginRepository: SqlPluginRepository) = {
    this(
      config,
      XLProduct.fromString(product),
      Version.fromPlatform(version),
      sqlPluginRepository
    )
    initProductSpecifics(XLProduct.fromString(product))
  }

  def this(config: PluginManagerProperties, product: String, version: String, sqlPluginRepository: SqlPluginRepository) = {
    this(
      config,
      XLProduct.fromString(product),
      Version.fromString(version).getOrElse(throw new IllegalArgumentException(s"Wrong version format: $version")),
      sqlPluginRepository
    )
    initProductSpecifics(XLProduct.fromString(product))
  }

  protected implicit val productConfig: ProductConfig = ProductConfig(product, version.copy(extra = None))

  override val repositories: mutable.Map[String, PluginsRepository] =
    mutable.Map.empty[String, PluginsRepository] ++ {
      PluginService.configuredRepositories(config).flatMap {
        case Success(repo) => List(repo.name -> repo)
        case Failure(ex) =>
          logger.warn(ex.getMessage)
          List.empty
      }.toMap
    }

  override def addRepository(repository: PluginsRepository): Boolean = {
    repositories.get(repository.name).map(_ => false).getOrElse {
      logger.info(s"Adding new repository; '${repository.name}' -> $repository")
      this.repositories += repository.name -> repository
      this.pluginManagerInstance = updatePluginManager()
      true
    }
  }

  override def deleteRepository(name: String): Boolean = {
    repositories.get(name).exists { _ =>
      logger.info(s"Removing repository '$name'")
      repositories -= name
      this.pluginManagerInstance = updatePluginManager()
      true
    }
  }

  def initProductSpecifics(xlproduct: XLProduct): Unit = {
    ConfigWrapper.initWith(xlproduct)
  }

  override def pluginManager: PluginManager = pluginManagerInstance

  private var pluginManagerInstance: PluginManager = updatePluginManager()

  private def updatePluginManager(): LocalPluginManager = {
    val nameGroupIdMap = repositories.collect {
      case (name, NexusPluginRepository(_, NexusRepositoryConfig(_, _, _, _, ListMethod.ByGroupId(groupId, _, _)), _)) =>
        name -> groupId
    }.toMap

    LocalPluginManagerFactory.createInstance(nameGroupIdMap, sqlPluginRepository)
  }

}
