package com.xebialabs.plugin.manager.startup

import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.plugin.manager.metadata.XLProduct
import com.xebialabs.plugin.manager.model.{DbPlugin, FilePlugin}
import com.xebialabs.plugin.manager.repository.sql.SqlPluginRepository
import com.xebialabs.plugin.manager.rest.api.PluginSource.LOCAL
import com.xebialabs.plugin.manager.rest.api.{PluginSource, PluginStatus}
import com.xebialabs.plugin.manager.util.PluginsTraverser.traverse
import com.xebialabs.xlplatform.sugar.PathSugar.path2File
import grizzled.slf4j.Logging
import org.apache.commons.io.FileUtils

import java.io.File
import java.nio.file.{Path, Paths}

object SourceOfTruth extends Enumeration {

  val FILE_SYSTEM: SourceOfTruth.Value = Value("filesystem")
  val DATABASE: SourceOfTruth.Value = Value("database")

}

/**
 * Reads plugins from DB and writes them onto the file system. Intended to be run on every product restart and cluster
 * node addition.
 */
class PluginSynchronizer(pluginRepository: SqlPluginRepository, product: String, pluginsDir: Path, sourceOfTruth: SourceOfTruth.Value) extends Logging {

  def this(pluginRepository: SqlPluginRepository, product: String, sourceOfTruth: SourceOfTruth.Value) {
    this(pluginRepository, product, Paths.get("plugins"), sourceOfTruth)
  }

  val xlProduct: XLProduct = XLProduct.fromString(product)
  initProductSpecifics(xlProduct)

  def syncPlugins(): Unit = {
    if (pluginsDir.exists && pluginRepository.pluginTablesExist) {
      info(s"Starting plugin synchronization, using $sourceOfTruth as source of truth")
      val dbPluginsBySource = pluginRepository.getAllWithBytes.groupBy(_.source.toString)

      traverse(pluginsDir, ConfigWrapper.extension)(
        (pluginSource, plugins) => syncPlugins(plugins, dbPluginsBySource.getOrElse(pluginSource.toString, Seq.empty), pluginSource),
        plugins => syncPlugins(plugins, dbPluginsBySource.getOrElse(LOCAL.toString, Seq.empty), LOCAL)
      )

      pluginRepository.updateAllPluginsStatusTo(PluginStatus.INSTALLED)
    } else {
      info(s"Skipping plugin synchronization, database is not ready yet.")
    }
  }

  private def syncPlugins(fsPlugins: List[FilePlugin], dbPlugins: Seq[DbPlugin], pluginSource: PluginSource.Value): Unit = {
    sourceOfTruth match {
      case SourceOfTruth.FILE_SYSTEM => syncUsingFsAsSourceOfTruth(fsPlugins, dbPlugins, pluginSource)
      case SourceOfTruth.DATABASE => syncUsingDbAsSourceOfTruth(fsPlugins, dbPlugins)
    }
  }

  def syncUsingDbAsSourceOfTruth(fsPlugins: List[FilePlugin], dbPlugins: Seq[DbPlugin]): Unit = {
    // delete what's on the FS and not in the DB
    fsPlugins.filter(fsPlugin => fsPlugin.doesntExistIn(dbPlugins)).foreach(cleanupFromFilesystem)

    // insert what's in the DB and not on the FS
    dbPlugins.filter(dbPlugin => dbPlugin.doesntExistIn(fsPlugins)).foreach(writeToFilesystem)

    // overwrite all db plugins that have a name match but version or content mismatch
    val mismatchedDbPlugins = dbPlugins.filter(dbPlugin => dbPlugin.hasDifferentContentOrVersionThanANameMatchIn(fsPlugins))
    val mismatchedFsPlugins = fsPlugins.filter(fsPlugin => fsPlugin.hasDifferentContentAndVersionThanAMatchIn(dbPlugins))

    mismatchedFsPlugins.foreach(cleanupFromFilesystem)
    mismatchedDbPlugins.foreach(writeToFilesystem)
  }

  def syncUsingFsAsSourceOfTruth(fsPlugins: List[FilePlugin], dbPlugins: Seq[DbPlugin], pluginSource: PluginSource.Value): Unit = {
    // delete what's in the DB and not on the FS
    dbPlugins.filter(dbPlugin => dbPlugin.doesntExistIn(fsPlugins)).foreach(cleanupFromDatabase)

    // insert what's on the FS and not in the DB
    fsPlugins.filter(fsPlugin => fsPlugin.doesntExistIn(dbPlugins)).foreach(fsPlugin => writeToDatabase(fsPlugin, pluginSource))

    // overwrite all db plugins that have a name match but version or content mismatch
    val mismatchedDbPlugins = dbPlugins.filter(dbPlugin => dbPlugin.hasDifferentContentOrVersionThanANameMatchIn(fsPlugins))
    val mismatchedFsPlugins = fsPlugins.filter(fsPlugin => fsPlugin.hasDifferentContentAndVersionThanAMatchIn(dbPlugins))

    mismatchedDbPlugins.foreach(cleanupFromDatabase)
    mismatchedFsPlugins.foreach(fsPlugin => writeToDatabase(fsPlugin, pluginSource))
  }

  private def writeToFilesystem(dbPlugin: DbPlugin): Unit = {
    info(s"Copying plugin ${dbPlugin.name} from database to filesystem")
    val fileName = s"${dbPlugin.name}${if (dbPlugin.version.isEmpty) "" else "-" + dbPlugin.version.get}.${dbPlugin.extension}"
    val targetFile = new File(s"$pluginsDir${File.separator}${dbPlugin.source}${File.separator}$fileName")
    FileUtils.writeByteArrayToFile(targetFile, dbPlugin.bytes.get)
  }

  private def cleanupFromFilesystem(fsPlugin: FilePlugin): Unit = {
    info(s"Removing plugin ${fsPlugin.name} from filesystem")
    FileUtils.forceDelete(fsPlugin.filePath)
  }

  private def cleanupFromDatabase(dbPlugin: DbPlugin): Unit = {
    info(s"Removing plugin ${dbPlugin.name} from database")
    pluginRepository.delete(dbPlugin)
  }

  private def writeToDatabase(plugin: FilePlugin, pluginSource: PluginSource.Value): Unit = {
    info(s"Inserting plugin ${plugin.name} into database")

    val dbPlugin = pluginSource match {
      case LOCAL => plugin.toLocalDbPlugin
      case _ => plugin.toOfficialDbPlugin(xlProduct)
    }

    pluginRepository.insert(dbPlugin)
  }

  private def initProductSpecifics(product: XLProduct): Unit = {
    ConfigWrapper.initWith(product)
  }
}
