package com.xebialabs.plugin.manager.validator

import scala.collection.mutable.ArrayBuffer


sealed trait TypeSystemValidationResult {

  def isSuccess: Boolean

  def requiresRestart: Boolean

  def restartReason: String

  def failureReason: String

}

case class TypeSystemValidationResults() {
  private val results: ArrayBuffer[TypeSystemValidationResult] = ArrayBuffer()

  def isFailure: Boolean = {
    !isSuccess
  }

  def isSuccess: Boolean = {
    results.forall(_.isSuccess)
  }

  def requireRestart: Boolean = {
    results.exists(_.requiresRestart)
  }

  def add(validationResult: TypeSystemValidationResult): TypeSystemValidationResults = {
    results += validationResult
    this
  }

  def restartReason: String = {
    results.collect {
      case r => r.restartReason
    }.mkString(",\n")
  }

  def failureReason: String = {
    results.collect {
      case r => r.failureReason
    }.mkString(",\n")
  }
}

sealed trait FailedTypeSystemValidation extends TypeSystemValidationResult {
  override def isSuccess: Boolean = false

  override def requiresRestart: Boolean = false
}

sealed trait SuccessfulTypeSystemValidation extends TypeSystemValidationResult {
  override def isSuccess: Boolean = true

  override def failureReason: String = ""
}

case class RestartNeeded(restartReason: String) extends SuccessfulTypeSystemValidation {
  val requiresRestart = true
}

case class RestartNotNeeded(restartReason: String = "") extends SuccessfulTypeSystemValidation {
  val requiresRestart = false
}

case class UnableToCreateTypeSystem(failureReason: String, restartReason: String = "") extends FailedTypeSystemValidation

case class UnsupportedPluginType(failureReason: String, restartReason: String = "") extends FailedTypeSystemValidation
