package com.xebialabs.plugin.manager.compatibility

import com.xebialabs.plugin.manager.model.{DbPlugin, FilePlugin}
import com.xebialabs.plugin.manager.validator.ValidatorUtils
import com.xebialabs.xlplatform.sugar.PathSugar.path2File
import grizzled.slf4j.Logging
import org.apache.commons.io.FileUtils
import org.jboss.windup.decompiler.api.{DecompilationFailure, DecompilationListener, DecompilationResult}
import org.jboss.windup.decompiler.fernflower.FernflowerDecompiler
import org.jboss.windup.decompiler.util.Filter
import org.jboss.windup.util.exception.WindupStopException

import java.io.File
import java.nio.file.{Files, Path, Paths, StandardCopyOption}
import java.util
import java.util.zip.{ZipEntry, ZipFile}
import scala.io.Source
import scala.jdk.CollectionConverters.EnumerationHasAsScala
import scala.util.Using

/**
  * This class is responsible for verifying if a plugin is compatible for deployment based on a set of
  * compatibility criteria. Currently this class is used for JDK17 compatibility but can be extended in the
  * future if there are other compatibility checks required.
  */
object Jdk17Compatibility extends Logging {

  val EXTENSIONS: List[String] = List("java", "py")

  val EXCLUSION_LIST: List[String] = List("javax.activation",
    "javax.batch",
    "javax.ejb",
    "javax.el",
    "javax.enterprise",
    "javax.faces",
    "javax.jms",
    "javax.json",
    "javax.jws",
    "javax.mail",
    "javax.persistence",
    "javax.resource",
    "javax.servlet",
    "javax.validation",
    "javax.websocket",
    "javax.ws",
    "javax.xml.bind")

  /**
    * Accepts a FilePlugin and determines if it meets the compatibility requirements for JDK17 deployment
    *
    * @param plugin - FilePlugin to check compatibility
    * @return Boolean - True = compatible, False = Incompatible
    */
  def isCompatible(plugin: FilePlugin): Boolean = {
    verifyPluginCompatibility(plugin.name, plugin.filePath)
  }

  /**
    * Accepts a DBPlugin and determines if it meets the compatibility requirements for JDK17 deployment. The DBPlugin must have plugin.bytes populated
    *
    * @param plugin - DBPlugin to check compatibility
    * @return Boolean - True = compatible, False = Incompatible
    */
  def isCompatible(plugin: DbPlugin): Boolean = {
    isCompatible(plugin.name, plugin.bytes.get)
  }

  /**
    * Accepts a Plugin name and Byte array representing the plugin and determines if it meets the compatibility requirements for JDK17 deployment.
    *
    * @param plugin - Name of the plugin
    * @param bytes  - Array of Bytes of the plugin being checked
    * @return Boolean - True = compatible, False = Incompatible
    */
  def isCompatible(plugin: String, bytes: Array[Byte]): Boolean = {
    val pluginFile = ValidatorUtils.createFileInWorkFolderFrom(bytes, plugin)
    try {
      verifyPluginCompatibility(plugin, pluginFile)
    } finally {
      try {
        FileUtils.deleteDirectory(pluginFile.getParentFile)
      } catch {
        case _: Exception =>
      }
    }
  }

  private def verifyPluginCompatibility(pluginName: String, pluginPath: File): Boolean = {
    val workFolder = Paths.get("work" + File.separator + pluginName + "-compatibilityCheck")
    try {
      workFolder.mkdirs()
      val fileFolderMap = extractPlugin(pluginPath, workFolder)
      val compatibleJars = fileFolderMap.takeWhile(e => {
        decompilePlugin(e._1, e._2)
        validatePluginSource(FileUtils.listFiles(e._2, Jdk17Compatibility.EXTENSIONS.toArray, true))
      })
      compatibleJars.size.equals(fileFolderMap.size)
    } catch {
      case e: Exception =>
        logger.error(s"Unable to verify JDK17 compatibility for plugin $pluginName: ${e.getMessage}, cause: ${e.getCause}")
        throw e
    } finally {
      try {
        FileUtils.deleteDirectory(workFolder)
      } catch {
        case _: Exception =>
      }
    }
  }

  def archiveContainsJars(zipFile: ZipFile): Boolean = zipFile.entries().asScala.exists(_.getName.endsWith(".jar"))

  private def extractPlugin(pluginPath: File, workFolder: Path): Map[File, Path] = {
    unzip(pluginPath, workFolder) // only used for python plugins, decompiler doesn't need .class files, jar path is enough

    if (archiveContainsJars(new ZipFile(pluginPath))) {
      val jars = workFolder.listFiles().filter(f => f.isFile && f.getName.endsWith(".jar"))
      jars.flatMap(jar => {
        val jarFolderPathAsString = jar.getAbsolutePath.substring(0, jar.getAbsolutePath.lastIndexOf(".jar"))
        val jarFolder = Paths.get(jarFolderPathAsString)
        jarFolder.mkdir()
        Map(jar -> jarFolder)
      }).toMap
    } else {
      Map(pluginPath -> workFolder)
    }
  }

  private def unzip(pluginFile: File, targetFolder: Path) = {
    val zipFile = new ZipFile(pluginFile)
    for (entry <- zipFile.entries.asScala) {
      val path = targetFolder.resolve(entry.getName)
      if (entry.isDirectory) {
        Files.createDirectories(path)
      } else {
        Files.createDirectories(path.getParent)
        Files.copy(zipFile.getInputStream(entry), path, StandardCopyOption.REPLACE_EXISTING)
      }
    }
  }

  private def decompilePlugin(plugin: File, workFolder: Path): Unit = {
    val decompiler = new FernflowerDecompiler()
    val decompileResult = new DecompilationResult
    val decompileListener = new DecompilationListener() {
      private var cancelled = false

      override
      def fileDecompiled(inputPath: util.List[String], outputPath: String): Unit = {
        try decompileResult.addDecompiled(inputPath, outputPath)
        catch {
          case stop: WindupStopException =>
            this.cancelled = true
            throw new WindupStopException(stop)
        }
      }

      override
      def decompilationFailed(inputPath: util.List[String], message: String): Unit = {
        decompileResult.addFailure(new DecompilationFailure(message, inputPath, null))
      }

      override
      def decompilationProcessComplete(): Unit = {}

      override
      def isCancelled: Boolean = this.cancelled
    }

    val filter = new Filter[ZipEntry] {
      override def decide(entry: ZipEntry): Filter.Result = if (entry.getName.endsWith("module-info.class")) {
        Filter.Result.REJECT
      } else {
        Filter.Result.ACCEPT
      }
    }

    val decompositionResult = decompiler.decompileArchive(plugin.toPath, workFolder, filter, decompileListener)
    if (!decompositionResult.getFailures.isEmpty) {
      logger.warn(s"Unable to decompile all files in plugin ${plugin.getName}.")
    }
  }

  private def validatePluginSource(fileList: util.Collection[File]): Boolean = {
    fileList.forEach(file => {
      Using(Source.fromFile(file)) { source =>
        val contents = source.getLines().mkString("\n")
        if (Jdk17Compatibility.EXCLUSION_LIST.exists(contents.contains)) {
          return false
        }
      }
    })
    true
  }

}
