package com.xebialabs.plugin.manager.repository.artifactory

import com.xebialabs.plugin.manager.repository.artifactory.ArtifactoryPluginRepositoryFixturesException.{CreateRepositoryError, DeleteError, UploadError}
import com.xebialabs.plugin.manager.repository.protocol.ArtifactMetadata
import com.xebialabs.plugin.manager.repository.{EntityOps, RespSyntax}
import grizzled.slf4j.Logging
import org.apache.pekko.actor.ActorSystem
import org.apache.pekko.http.scaladsl.model.MediaRanges.{`*/*`, `text/*`}
import org.apache.pekko.http.scaladsl.model.MediaTypes.{`application/json`, `application/octet-stream`}
import org.apache.pekko.http.scaladsl.model.Multipart.FormData
import org.apache.pekko.http.scaladsl.model._
import org.apache.pekko.http.scaladsl.model.headers.{Accept, Authorization}
import org.apache.pekko.stream.Materializer
import org.apache.pekko.stream.scaladsl.StreamConverters
import org.apache.pekko.util.ByteString

import scala.concurrent.{ExecutionContext, Future}


class ArtifactoryRepositoryManager(name: String, server: ArtifactoryServerConfig)(implicit m: Materializer, ec: ExecutionContext, as: ActorSystem)
  extends ArtifactoryJsonProtocol with Logging {

  def addRepository(repository: String): Future[Unit] = {
    logger.info(s"$name: Creating repository $repository")
    for {
      response <- createRepository(repository)
      _ <- response
        .onSuccess(_.asString)
        .onFailure((status, msg) => CreateRepositoryError(repository, status, msg))
    } yield {
      logger.info(s"$name: Created repository to ${server.uri}: $repository")
    }
  }

  def addArtifacts(repository: String, artifacts: ArtifactoryArtifact*): Future[Seq[ArtifactMetadata]] =
    Future.sequence(artifacts.map(a => addArtifact(repository, a)))

  def addJsonArtifacts(repository: String, artifacts: ArtifactoryArtifact*): Future[Seq[ArtifactMetadata]] =
    Future.sequence(artifacts.map(a => addJsonArtifact(repository, a)))

  def addJsonArtifact(repository: String, artifact: ArtifactoryArtifact): Future[ArtifactMetadata] =
    addArtifact(repository, artifact, Some(`application/json`))

  def addArtifact(repository: String, artifact: ArtifactoryArtifact, explicitContentType: Option[ContentType] = None): Future[ArtifactMetadata] = {
    val entity = uploadEntity(artifact, explicitContentType)

    logger.info(s"$name: Uploading artifact ${artifact.metadata.id}")

    val path = s"/$repository/${createPath(artifact.metadata)}"

    for {
      response <- uploadArtifact(path, entity)
      artifactMetadata <- response
        .onSuccess(_.asJson[ArtifactMetadata])
        .onFailure((status, msg) => UploadError(artifact.metadata, status, msg))
    } yield {
      logger.info(s"$name: Uploaded artifact to ${server.uri}: ${artifact.metadata.id}")
      artifactMetadata
    }
  }

  private def createPath(metadata: ArtifactMetadata): String = {
    val groupPath = metadata.groupId.replace('.', '/')
    val artifactVersion = s"${metadata.artifactId}-${metadata.version.id}.${metadata.packaging}"
    s"$groupPath/${metadata.artifactId}/${metadata.version.id}/$artifactVersion"
  }

  def removeArtifact(repository: String, artifact: ArtifactMetadata): Future[Unit] = {
    logger.trace(s"$name: removeArtifact($repository, $artifact)}")
    val artifactPath = s"/$repository/${createPath(artifact.metadata)}"
    for {
      resp <- deleteArtifact(artifactPath)
      _ <- resp
        .onSuccess(_.asString)
        .onFailure((status, msg) => DeleteError(artifact, status, msg))
    } yield {
      logger.trace(s"$name: removed artifact ${artifact.id} from '$repository' repository.")
    }
  }

  private def createRepository(repository: String): Future[HttpResponse] = {
    val uri = server.uri.withPath(server.serviceUri.path / "api" / "repositories" / repository)
    server.httpRequest(
      HttpRequest(HttpMethods.PUT, uri = uri, headers = List(
        Authorization(server.credentials),
        Accept(`*/*`),
      ))
    )
  }

  private def uploadEntity(artifact: ArtifactoryArtifact, explicitContentType: Option[ContentType] = None): RequestEntity = {
    val metadata = artifact.metadata

    val contentType: ContentType = explicitContentType.getOrElse(`application/octet-stream`)

    val content: FormData.BodyPart = artifact.content match {
      case Left(path) =>
        Multipart.FormData.BodyPart.fromPath("file", contentType, path)

      case Right(Left(input)) =>
        Multipart.FormData.BodyPart("file",
          HttpEntity.IndefiniteLength(contentType, StreamConverters.fromInputStream(() => input)),
          Map("filename" -> metadata.fileName)
        )

      case Right(Right(data)) =>
        Multipart.FormData.BodyPart("file",
          HttpEntity(contentType, ByteString(data.getBytes)),
          Map("filename" -> metadata.fileName)
        )

    }
    Multipart.FormData(content).toEntity()
  }

  protected def uploadArtifact(path: String, entity: RequestEntity): Future[HttpResponse] = {
    val uri = server.uri.withPath(server.serviceUri.path ++ Uri.Path(path))
    server.httpRequest(
      HttpRequest(HttpMethods.PUT, uri = uri, entity = entity, headers = List(
        Authorization(server.credentials),
        Accept(`*/*`),
      ))
    )
  }


  private def deleteArtifact(path: String): Future[HttpResponse] = {
    val uri = server.uri.withPath(server.serviceUri.path ++ Uri.Path(path))
    for {
      response <- server.httpRequest(
        HttpRequest(HttpMethods.DELETE, uri = uri, headers = List(
          Authorization(server.credentials),
          Accept(`text/*`)
        ))
      )
    } yield response
  }
}


sealed abstract class ArtifactoryPluginRepositoryFixturesException(message: String) extends Exception(message)

object ArtifactoryPluginRepositoryFixturesException {

  case class CreateRepositoryError(repository: String, status: StatusCode, message: String) extends ArtifactoryPluginRepositoryFixturesException(
    s"Error creating repository $repository: $status: $message"
  )

  case class UploadError(metadata: ArtifactMetadata, status: StatusCode, message: String) extends ArtifactoryPluginRepositoryFixturesException(
    s"Error uploading artifact ${metadata.id}: $status: $message"
  )

  case class DeleteError(metadata: ArtifactMetadata, status: StatusCode, message: String) extends ArtifactoryPluginRepositoryFixturesException(
    s"Error deleting artifact ${metadata.id}: $status: $message"
  )

}
