package com.xebialabs.plugin.manager

import com.xebialabs.plugin.manager.exception.PluginRepositoryException.{ChecksumMismatch, ContentLengthMismatch}
import com.xebialabs.plugin.manager.metadata.Version
import org.apache.commons.codec.digest.DigestUtils
import org.apache.pekko.http.scaladsl.model.{HttpEntity, HttpResponse, Uri}
import org.apache.pekko.stream.Materializer
import org.apache.pekko.stream.scaladsl.Sink
import spray.json.{JsonReader, _}

import java.io.ByteArrayOutputStream
import java.nio.charset.StandardCharsets.UTF_8
import scala.concurrent.{ExecutionContext, Future}
import scala.util.{Failure, Success, Try}

package object repository {

  def paramsToQuery(params: Map[String, Option[String]]): Uri.Query = Uri.Query(params.collect { case (k, Some(v)) => k -> v })

  implicit class EntityOps(val entity: HttpEntity) extends AnyVal {
    def asByteArray(implicit m: Materializer, ec: ExecutionContext): Future[Array[Byte]] = {
      for {
        outStream <- entity.dataBytes.runWith(
          Sink.fold(new ByteArrayOutputStream()) { (out, bs) =>
            out.write(bs.toArray, 0, bs.length)
            out
          }
        )
      } yield {
        try {
          outStream.toByteArray
        } finally {
          outStream.close()
        }
      }
    }

    def asString(implicit m: Materializer, ex: ExecutionContext): Future[String] =
      entity.dataBytes
        .map(_.utf8String)
        .runWith(Sink.seq)
        .map(_.mkString(""))


    def asJson[T: JsonReader](implicit m: Materializer, ec: ExecutionContext): Future[T] =
      asString
        .map(_.parseJson.convertTo[T])
  }

  implicit class RespSyntax(val resp: HttpResponse) extends AnyVal {
    def onSuccess[A](f: HttpEntity => Future[A]): SuccessMapped[A] = {
      if (resp.status.isFailure()) {
        IsFailed(resp)
      } else {
        OnSuccess(f(resp.entity))
      }
    }
  }

  sealed trait SuccessMapped[+A] {
    def onFailure[E <: Throwable](err: (Int, String) => E)(implicit ec: ExecutionContext, m: Materializer): Future[A] = this match {
      case OnSuccess(resp) => resp
      case IsFailed(resp) => resp.entity.asString.flatMap { entity =>
        Future.failed(err(resp.status.intValue(), entity))
      }
    }
  }

  case class OnSuccess[A](value: Future[A]) extends SuccessMapped[A]

  case class IsFailed(resp: HttpResponse) extends SuccessMapped[Nothing]

  def verifyPlugin(pluginId: PluginId.Artifact, sha1: Array[Byte], plugin: Array[Byte], length: Option[Long]): Future[(String, Long)] = Future.fromTry {
    for {
      digest <- verifySHA1(pluginId, sha1, plugin)
      length <- verifyContentLength(pluginId, length, plugin.length)

    } yield (digest, length)
  }

  def latestVersion(pv: Version)
                   (versions: Seq[Version]): Option[Version] =
    versions
      .filter(isProductVersion(pv))
      .sortBy(_.id)
      .lastOption

  private def verifyContentLength(pluginId: PluginId.Artifact, expected: Option[Long], found: Long): Try[Long] = expected match {
    case None => Success(found)
    case Some(length) if found == length => Success(found)
    case Some(length) => Failure(ContentLengthMismatch(pluginId, found, length))
  }

  private def verifySHA1(pluginId: PluginId.Artifact, expected: Array[Byte], found: Array[Byte]): Try[String] =
    for {
      sha1 <- Try(new String(expected, UTF_8))
      digest <- Try(DigestUtils.sha1Hex(found))
      result <- {
        if (digest == sha1) {
          Success(digest)
        } else {
          Failure(ChecksumMismatch(pluginId, digest, sha1))
        }
      }
    } yield result

  private def isProductVersion(pv: Version)(v: Version): Boolean =
    pv.major == v.major && pv.minor == v.minor && pv.revision == v.revision

}
