package com.xebialabs.plugin.manager.rest.api

import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.plugin.manager.PluginId.{Artifact, LocalFile}
import com.xebialabs.plugin.manager.config.ConfigWrapper
import com.xebialabs.plugin.manager.metadata.{ArtifactId, PluginMetadata, Version}
import com.xebialabs.plugin.manager.repository.PluginsRepository
import com.xebialabs.plugin.manager.rest.dto.{PluginDto, PluginDtoList, RepositoryDto}
import com.xebialabs.plugin.manager.service.{InstallationFailure, InstallationResult, PluginService}
import com.xebialabs.plugin.manager.{Plugin, PluginId}
import org.apache.commons.io.IOUtils
import org.joda.time.format.DateTimeFormat
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller

import java.io.OutputStream
import java.util.Collections
import jakarta.servlet.http.{HttpServletRequest, Part}
import jakarta.ws.rs._
import jakarta.ws.rs.core.{MediaType, Response, StreamingOutput}
import scala.concurrent.duration._
import scala.concurrent.{Await, ExecutionContext, Future}
import scala.jdk.CollectionConverters._
import scala.language.postfixOps
import scala.reflect.io.File
import scala.util.{Failure, Success, Try}

@Controller
class PluginResourceImpl @Autowired()(val pluginService: PluginService) extends AbstractSecuredResource with PluginResource {

  val DATE_TIME_FORMAT = "MM-dd-YYYY hh:mm:ss z"
  implicit val ec: ExecutionContext = pluginService.pluginManager.ec

  override def list(): Response = {
    checkPermission(ADMIN)
    Response.ok(
      PluginDtoList(pluginService.extend(pluginService.listInstalled())),
      MediaType.APPLICATION_JSON
    ).build()
  }

  override def installOrUpdate(request: HttpServletRequest, pluginId: String): Response = {
    def install(pluginId: PluginId, filePart: Part): Response = {
      val bytes = getBytesFrom(filePart)
      pluginService.installOrUpdate(Plugin(pluginId, None, bytes))
      Response.ok().build()
    }

    def convertToLocalAndInstall(pluginId: String, filePart: Part): Response = {
      Try(LocalFile(pluginId, getBytesFrom(filePart))) match {
        case Success(localFile) => install(localFile, filePart)
          Response.ok().build()
        case Failure(exception) => Response
          .status(Response.Status.BAD_REQUEST)
          .entity(InstallationFailure(pluginId, exception.getMessage, Seq.empty, Seq.empty))
          .build()
      }
    }

    def convertAndInstall(pluginId: String, basename: String, version: String, filePart: Part): Response = {
      // first check if the plugin is found in metadata
      val searchResults = pluginService.searchOfficial(basename)
      // if it is, then it's an official plugin
      if (searchResults.size == 1) {
        val artifact = searchResults.head._1.toArtifact
        install(artifact.copy(artifactVersion = Version.fromString(version).orNull), filePart)
      } else {
        // if not, check if there is an official plugin with same name already installed in the system
        val installedOfficial = pluginService.listInstalled().collect { case a: Artifact => a }.filter(officialPlugin => officialPlugin.artifactId.equalsIgnoreCase(basename))
        // if there is, then it's an official plugin
        if (installedOfficial.size == 1) {
          val installed = installedOfficial.head
          val idString = installed.repository + ":" + installed.groupId + ":" + pluginId
          install(PluginId.Artifact.fromIdString(idString).get, filePart)
        } else {
          // if not, then it's a 3rd party plugin
          convertToLocalAndInstall(pluginId, filePart)
        }
      }
    }

    checkPermission(ADMIN)

    val filePart = validateAndGetPluginFile(request)

    pluginId match {
      case PluginId.patternVersionWithNightly(name, versionWithNightly) =>
        convertAndInstall(pluginId, name, versionWithNightly, filePart)
      case PluginId.pattern(name, version) =>
        convertAndInstall(pluginId, name, version, filePart)
      case _ =>
        convertToLocalAndInstall(pluginId, filePart)
    }
  }

  private def getBytesFrom(filePart: Part): Array[Byte] = {
    IOUtils.toByteArray(filePart.getInputStream)
  }


  override def install(request: HttpServletRequest, pluginId: String): Response = {
    checkPermission(ADMIN)
    val filePart = validateAndGetPluginFile(request)
    val bytes = getBytesFrom(filePart)
    val installationResult = pluginService.installOfficialOrLocal(pluginId, bytes, LocalFile(pluginId, bytes))
    toResponse(installationResult)
  }

  override def update(request: HttpServletRequest,
                      existingPluginName: String,
                      newPluginId: String): Response = {
    checkPermission(ADMIN)
    val filePart = validateAndGetPluginFile(request)
    val bytes = getBytesFrom(filePart)
    val installationResult = pluginService.update(existingPluginName, LocalFile(newPluginId, bytes), bytes)
    toResponse(installationResult)
  }

  private def toResponse(installationResult: InstallationResult): Response = {
    val status = if (installationResult.success) Response.Status.OK else Response.Status.BAD_REQUEST
    Response.status(status)
      .entity(installationResult)
      .build()
  }

  override def uninstall(repositoryId: String,
                         groupId: String,
                         artifactId: String,
                         version: String): Response = {
    checkPermission(ADMIN)

    val pluginSource = PluginSource.withName(repositoryId)
    val optionalVersion = if (version.isEmpty) None else Some(version.split("/")(1))

    pluginService.uninstall(pluginSource, groupId, artifactId, optionalVersion).map { _ =>
      Response.ok.build()
    }.fold(
      err => throw err,
      ok => ok
    )
  }

  override def search(query: String): Response = {
    checkPermission(ADMIN)
    Response.ok(
      PluginDtoList(pluginService.search(query).map(toPluginDto).toSeq),
      MediaType.APPLICATION_JSON
    ).build()
  }

  override def listRepositories(): Response = {
    checkPermission(ADMIN)
    Response.ok(RepositoryDto(pluginService.repositories.values.toSeq), MediaType.APPLICATION_JSON).build()
  }

  override def updateRepository(repositoryId: String): Response = {
    checkPermission(ADMIN)
    val lastDownloaded = withRepo(repositoryId)(_.update().map(DateTimeFormat.forPattern(DATE_TIME_FORMAT).print(_)))
    Response.ok(
      Collections.singletonMap("lastDownloaded", lastDownloaded)
    ).build()
  }

  override def listRepository(repositoryId: String): Response = {
    checkPermission(ADMIN)
    Response.ok(
      PluginDtoList(withRepo(repositoryId)(_.list()).map(toPluginDto).toSeq),
      MediaType.APPLICATION_JSON
    ).build()
  }

  override def installOrUpdateFromRepository(repositoryId: String,
                                             groupId: String,
                                             artifactId: String,
                                             version: String): Response = {
    checkPermission(ADMIN)
    val packaging = defaultRepositoryExtension(groupId)
    val artifact = PluginId.Artifact.apply(repositoryId, groupId, artifactId, version, packaging)
    pluginService.installOrUpdateFromRepository(artifact).map { _ =>
      Response.ok(artifact, MediaType.APPLICATION_JSON).build()
    }.fold(
      err => throw err,
      ok => ok
    )
  }

  override def installFromRepository(repositoryId: String,
                                     groupId: String,
                                     artifactId: String,
                                     version: String): Response = {
    val packaging = defaultRepositoryExtension(groupId)
    installFromRepository(repositoryId, groupId, artifactId, version, packaging)
  }

  override def installFromRepository(repositoryId: String,
                                     groupId: String,
                                     artifactId: String,
                                     version: String,
                                     packaging: String): Response = {
    checkPermission(ADMIN)
    val artifact = PluginId.Artifact.apply(repositoryId, groupId, artifactId, version, packaging)
    val installationResult = pluginService.installFromRepository(artifact)
    toResponse(installationResult)
  }

  override def updateFromRepository(repositoryId: String, groupId: String, artifactId: String): Response = {
    val packaging = defaultRepositoryExtension(groupId)
    updateFromRepository(repositoryId, groupId, artifactId, packaging)
  }

  override def updateFromRepository(repositoryId: String, groupId: String, artifactId: String, packaging: String): Response = {
    checkPermission(ADMIN)
    val artifact = PluginId.Artifact.apply(repositoryId, groupId, artifactId, version = Version.zero.id, packaging)
    val installationResult = pluginService.updateFromRepository(artifact)
    toResponse(installationResult)
  }

  override def getLogo(repositoryId: String, groupId: String, artifactId: String): Response = {
    checkPermission(ADMIN)
    pluginService.getLogo(repositoryId, groupId, artifactId).map { logoFile =>
      val input = File(logoFile).inputStream()
      val stream = new StreamingOutput {
        override def write(output: OutputStream): Unit = {
          output.write(IOUtils.toByteArray(input))
          input.close()
        }
      }
      Response.ok(stream, MediaType.APPLICATION_OCTET_STREAM).build()
    }.getOrElse {
      throw new NotFoundException(s"Couldn't find logo for '$groupId:$artifactId' in repository '$repositoryId'")
    }
  }

  private def withRepo[T](repositoryId: String, duration: Duration = 60 seconds)(action: PluginsRepository => Future[T]): T = {
    withRepo0(repositoryId) { repository =>
      Await.result(action(repository), duration)
    }
  }

  private def withRepo0[T](repositoryId: String)(action: PluginsRepository => T): T = {
    pluginService.repositories.get(repositoryId) match {
      case None =>
        throw new NotFoundException(s"Unknown plugin repository '$repositoryId'")

      case Some(repository) =>
        action(repository)
    }
  }

  private def defaultRepositoryExtension(groupId: String): String = {
    if (groupId == ConfigWrapper.GROUP_ID_XLR) {
      ConfigWrapper.EXTENSION_JAR
    } else {
      ConfigWrapper.EXTENSION_XLDP
    }
  }

  private def filterFilePart(part: Part): Boolean = {
    part.getSubmittedFileName != null && part.getSubmittedFileName.nonEmpty
  }

  private def validateAndGetPluginFile(request: HttpServletRequest): Part = {
    val inputParts = request.getParts.asScala.filter(filterFilePart).toList

    val fileCount = inputParts.size
    if (fileCount > 1) {
      throw new BadRequestException("Can not upload more than one file")
    } else if (fileCount == 0) {
      throw new BadRequestException("No plugin file provided")
    }

    inputParts.head
  }

  def toPluginDto: ((ArtifactId, Option[PluginMetadata])) => PluginDto = tuple => PluginDto(tuple._1.toPluginId(), tuple._2)

}
