package com.xebialabs.xlplatform.xlrepository

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlplatform.xlrepository.RepositoryCiCache.CachedDepth
import grizzled.slf4j.Logging

import scala.collection.mutable

object RepositoryCiCacheHolder extends Logging {
  private[this] val Holder = new ThreadLocal[Option[RepositoryCiCache]] {
    // Initial value is none, this will ensure that you will never by accident initialize a thread local cache
    override def initialValue(): Option[RepositoryCiCache] = None
  }

  def get(): RepositoryCiCache = Holder.get().getOrElse(new RepositoryCiCache("one-shot"))

  def newThreadCache(): RepositoryCiCache = {
    logger.debug("Creating new ThreadLocal repository CI cache")
    val repositoryCiCache = new RepositoryCiCache(Thread.currentThread().getName)
    Holder.set(Option(repositoryCiCache))
    repositoryCiCache
  }

  def clear(): Unit = {
    logger.debug(s"Removing ThreadLocal repository CI cache")
    Holder.remove()
  }
}

object RepositoryCiCache {
  type CachedDepth = Int
}

class RepositoryCiCache(name: String) extends Logging {
  private[this] val _cache = mutable.Map[String, CachedConfigurationItem]()

  def get(id: String): Option[CachedConfigurationItem] = _cache.get(id)
  def getForDepth(id: String, depth: CachedDepth): Option[ConfigurationItem] = get(id).flatMap(cci => if (cci.depth < depth) None else Some(cci.ci))

  def cache(ci: ConfigurationItem, depth: CachedDepth): Unit = {
    logger.trace(s"Caching [${ci.getId}] in repository cache [$name]")
    _cache.put(ci.getId, CachedConfigurationItem(ci, depth))
  }
}

case class CachedConfigurationItem(ci: ConfigurationItem, depth: CachedDepth)
