package com.xebialabs.deployit.repository;

import com.google.common.base.Function;
import com.google.common.collect.FluentIterable;
import com.xebialabs.deployit.engine.spi.command.*;
import com.xebialabs.deployit.engine.spi.command.util.Update;
import com.xebialabs.deployit.engine.spi.event.DeployitEventListener;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Application;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import nl.javadude.t2bus.Subscribe;

import java.util.List;

@DeployitEventListener
public class RepositoryEventListener {

    @Subscribe
    public void receiveCreate(CreateCiCommand command) {
        checkApplicationOnUniqueness(command.getCi());
        RepositoryServiceHolder.getRepositoryService().create(command.getCi());
    }

    @Subscribe
    public void receiveCreate(CreateCisCommand command) {
        final List<ConfigurationItem> listOfCis = command.getCis();
        ConfigurationItem[] cis = listOfCis.toArray(new ConfigurationItem[listOfCis.size()]);
        checkApplicationOnUniqueness(cis);
        RepositoryServiceHolder.getRepositoryService().create(cis);
    }

    @Subscribe
    public void receiveUpdate(UpdateCiCommand command) {
        RepositoryServiceHolder.getRepositoryService().update(command.getUpdate().getNewCi());
    }

    @Subscribe
    public void receiveUpdate(UpdateCisCommand command) {
        List<Update> updates = command.getUpdates();
        FluentIterable<ConfigurationItem> transform = FluentIterable.from(updates).transform(new Function<Update, ConfigurationItem>() {
            @Override
            public ConfigurationItem apply(final Update input) {
                return input.getNewCi();
            }
        });
        ConfigurationItem[] cis = transform.toArray(ConfigurationItem.class);

        RepositoryServiceHolder.getRepositoryService().createOrUpdate(cis);
    }

    @Subscribe
    public void receiveDelete(DeleteCiCommand command) {
        RepositoryServiceHolder.getRepositoryService().delete(command.getCiId());
    }

    @Subscribe
    public void receiveDelete(DeleteCisCommand command) {
        RepositoryServiceHolder.getRepositoryService().delete(command.getCis().toArray(new String[command.getCis().size()]));
    }

    @Subscribe
    public void receiveMove(MoveCiCommand command) {
        RepositoryServiceHolder.getRepositoryService().move(command.getId(), command.getTargetId());
    }

    @Subscribe
    public void receiveCopy(CopyCiCommand command) {
        checkApplicationOnUniquenessByName(getName(command.getTargetId()), command.getType());
        RepositoryServiceHolder.getRepositoryService().copy(command.getId(), command.getTargetId());
    }

    @Subscribe
    public void receiveRename(RenameCiCommand command) {
        checkApplicationOnUniquenessByName(command.getTargetName(), command.getType());
        RepositoryServiceHolder.getRepositoryService().rename(command.getId(), command.getTargetName());
    }

    private void checkApplicationOnUniqueness(ConfigurationItem... cis) {
        for (ConfigurationItem ci : cis) {
            checkApplicationOnUniquenessByName(ci.getName(), ci.getType());
        }
    }

    private void checkApplicationOnUniquenessByName(String ciName, Type type) {
        Type applicationType = Type.valueOf(Application.class);
        if (!applicationType.instanceOf(type)) {
            return;
        }

        SearchParameters parameters = new SearchParameters().setType(applicationType).setName(ciName);
        if (!RepositoryServiceHolder.getRepositoryService().list(parameters).isEmpty()) {
            throw new ItemAlreadyExistsException("Application with name [%s] already exists", ciName);
        }
    }

    private String getName(String id) {
        int indexOfLastSlash = id.lastIndexOf('/');
        if (indexOfLastSlash > -1) {
            return id.substring(indexOfLastSlash + 1);
        }
        return id;
    }
}
