package com.xebialabs.deployit.jcr;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.jcr.*;

@Component
public class JcrTemplate {

    private final Repository repository;
    private final Credentials credentials;

    private static final ThreadLocal<Session> SESSION_STORE = new ThreadLocal<Session>();
    private static final Logger logger = LoggerFactory.getLogger(JcrTemplate.class);

    @Autowired
    public JcrTemplate(final Repository repository, Credentials credentials) {
        this.repository = repository;
        this.credentials = credentials;
    }

    public final <T> T execute(final JcrCallback<T> callback) throws RuntimeRepositoryException {
            return this.<T>execute(callback, false);
    }

    public final <T> T execute(final JcrCallback<T> callback, final boolean createNewSession) throws RuntimeRepositoryException {
        Session session = SESSION_STORE.get();
        boolean newSession = createNewSession;
        if (!newSession) {
            if (session == null) {
                logger.debug("Did not find an existing session, creating a session for one JCR request.");
                newSession = true;
            } else if (!session.isLive()) {
                logger.debug("Existing session is no longer live, creating a session for one JCR request.");
                newSession = true;
            }
        }

        try {
            if (newSession) {
                session = getSession();
            }
            return callback.doInJcr(session);
        } catch (RepositoryException e) {
            throw new RuntimeRepositoryException(e.getMessage(), e);
        } finally {
            if (newSession) {
                //noinspection ConstantConditions
                session.logout();
            }
        }
    }

    private Session getSession() {
        try {
            return repository.login(credentials);
        } catch (AccessDeniedException ade) {
            throw new RuntimeRepositoryException("Failed to complete your request.", ade);
        } catch (LoginException le) {
            throw new RuntimeRepositoryException("Couldn't login to the repository", le);
        } catch (RepositoryException e) {
            throw new RuntimeRepositoryException(e.getMessage(), e);
        }
    }

    public void login() {
        SESSION_STORE.set(getSession());
    }

    public boolean isLoggedIn() {
        return SESSION_STORE.get() != null && SESSION_STORE.get().isLive();
    }

    public void logout() {
        if (SESSION_STORE.get() != null) {
            SESSION_STORE.get().logout();
            SESSION_STORE.remove();
        } else {
            logger.warn("Superfluous logout call found.", new RuntimeException());
        }
    }

    public Repository getRepository() {
        return repository;
    }
}
