package com.xebialabs.deployit.repository;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.util.*;
import javax.jcr.*;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Collections2;
import com.google.common.collect.Maps;
import com.google.common.io.ByteSource;
import com.google.common.io.ByteStreams;

import com.xebialabs.deployit.booter.local.utils.Closeables;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.core.*;
import com.xebialabs.deployit.engine.spi.artifact.resolution.ResolvedArtifactFile;
import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.io.Exploder;
import com.xebialabs.deployit.io.Imploder;
import com.xebialabs.deployit.jcr.JcrUtils;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.FolderArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.security.Permissions2;
import com.xebialabs.deployit.util.DevNull;
import com.xebialabs.deployit.util.JavaCryptoUtils;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.xlplatform.artifact.resolution.ArtifactResolverRegistry;
import com.xebialabs.xlplatform.artifact.resolution.jcr.JcrArtifactResolver;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newLinkedHashMap;
import static com.xebialabs.deployit.booter.local.utils.Strings.isBlank;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CREATED_AT_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CREATED_BY_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.DATA_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.FILENAME_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.LAST_MODIFIED_AT_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.LAST_MODIFIED_BY_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.TOKEN_PROPERTY_NAME;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.STRING;
import static com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact.CHECKSUM_PROPERTY_NAME;
import static com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact.FILE_URI_PROPERTY_NAME;
import static com.xebialabs.deployit.repository.JcrPathHelper.getAbsolutePathFromId;
import static com.xebialabs.deployit.repository.JcrPathHelper.getIdFromAbsolutePath;
import static com.xebialabs.deployit.repository.NodeUtils.findContainmentListProperty;
import static com.xebialabs.deployit.repository.NodeUtils.getReferencedCiNode;
import static com.xebialabs.deployit.repository.NodeUtils.typeOf;
import static javax.jcr.nodetype.NodeType.MIX_VERSIONABLE;

/**
 * Writes an {@link com.xebialabs.deployit.plugin.api.udm.ConfigurationItem} to a JCR {@link javax.jcr.Node}.
 */
class NodeWriter {

    private final Session session;
    private final ConfigurationItem item;
    private final Node node;
    private final PasswordEncrypter passwordEncrypter;
    private boolean basicsWritten = false;

    NodeWriter(Session session, ConfigurationItem item, Node node, PasswordEncrypter passwordEncrypter) {
        this.session = checkNotNull(session);
        this.item = checkNotNull(item);
        this.node = checkNotNull(node);
        this.passwordEncrypter = checkNotNull(passwordEncrypter);
    }

    void writeBasics() throws RepositoryException {
        node.setProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME, item.getType().toString());
        if (item instanceof BaseConfigurationItem) {
            String token = ((BaseConfigurationItem) item).get$token();
            if (Strings.isNullOrEmpty(token)
                    || (node.hasProperty(TOKEN_PROPERTY_NAME) && node.getProperty(TOKEN_PROPERTY_NAME).getString().equals(token))) {
                node.setProperty(TOKEN_PROPERTY_NAME, UUID.randomUUID().toString());
            } else {
                throw new ItemConflictException("Repository entity [%s] has been updated since you read it. Please reload the CI from the repository again.", item.getId());
            }
        }
        basicsWritten = true;
    }

    void write() throws RepositoryException {
        if (!basicsWritten) {
            writeBasics();
        }
        // First copy the data then the properties, checksum + fileUri is possibly set in the copyData call.
        setMixinsCorrect();
        copyData();
        copyValuesIntoNode();
        copyMetadata();
        updateParentForListAsContainmentRelation();
    }

    private void setMixinsCorrect() throws RepositoryException {
        if (item.getType().getDescriptor().isVersioned()) {
            if (node.canAddMixin(MIX_VERSIONABLE)) {
                node.addMixin(MIX_VERSIONABLE);
            }
        } else if (node.isNodeType(MIX_VERSIONABLE)) {
            node.removeMixin(MIX_VERSIONABLE);
        }
    }

    private void copyData() throws RepositoryException {
        if (item instanceof SourceArtifact) {
            SourceArtifact artifact = (SourceArtifact) item;
            MessageDigest sha1 = JavaCryptoUtils.getSha1();
            OverthereFile file = ((Artifact) item).getFile();
            if (file != null) {
                node.setProperty(FILENAME_PROPERTY_NAME, file.getName());
                InputStream dataInput;
                if (item instanceof FolderArtifact) {
                    if (file.isDirectory()) {
                        try {
                            ByteSource implode = Imploder.implode(file, sha1);
                            dataInput = implode.openStream();
                        } catch (IOException e) {
                            throw new RuntimeIOException("Could not zip up the data in file " + file, e);
                        }
                    } else {
                        try {
                            Exploder.calculateCheckSum(newInputStreamSupplier(file), sha1);
                        } catch (IOException e) {
                            throw new RuntimeIOException("Could not calculate checksum for zipped Folder artifact.", e);
                        }
                        dataInput = file.getInputStream();
                    }
                } else {
                    dataInput = new DigestInputStream(file.getInputStream(), sha1);
                }

                writeBinaryData(dataInput);
                if (item.hasProperty(CHECKSUM_PROPERTY_NAME) && isBlank((String) item.getProperty(CHECKSUM_PROPERTY_NAME))) {
                    // We're setting the checksum on the item here, instead of storing it in the repository, as we will copy the properties after this call.
                    item.setProperty(CHECKSUM_PROPERTY_NAME, JavaCryptoUtils.digest(sha1));
                }

                String fileUri;
                try {
                    fileUri = JcrArtifactResolver.Protocol() + ":" + URLEncoder.encode(file.getName(), "UTF-8");
                } catch (UnsupportedEncodingException e) {
                    throw new RuntimeIOException("Could not encode the file name.", e);
                }
                item.setProperty(SourceArtifact.FILE_URI_PROPERTY_NAME, fileUri);
            } else if (!node.hasProperty(FILENAME_PROPERTY_NAME)) {
                ResolvedArtifactFile resolvedArtifactFile = ArtifactResolverRegistry.resolve(artifact);
                try {
                    node.setProperty(FILENAME_PROPERTY_NAME, resolvedArtifactFile.getFileName());

                    if (item.hasProperty(CHECKSUM_PROPERTY_NAME) && isBlank((String) item.getProperty(CHECKSUM_PROPERTY_NAME))) {
                        try {
                            if (item instanceof FolderArtifact) {
                                Exploder.calculateCheckSum(newInputStreamSupplier(resolvedArtifactFile.openStream()), sha1);
                            } else {
                                InputStream dataInput = new DigestInputStream(resolvedArtifactFile.openStream(), sha1);
                                try {
                                    ByteStreams.copy(dataInput, new DevNull());
                                } finally {
                                    Closeables.closeQuietly(dataInput);
                                }
                            }
                        } catch (IOException e) {
                            throw new RuntimeIOException("Could not calculate checksum for the artifact.", e);
                        }
                        item.setProperty(CHECKSUM_PROPERTY_NAME, JavaCryptoUtils.digest(sha1));
                    }
                } finally {
                    try {
                        resolvedArtifactFile.close();
                    } catch (IOException e) {
                        logger.error("Error when closing the resolved file {}", ((SourceArtifact) item).getFileUri());
                        logger.error("Exception was: ", e);
                    }
                }
            } else {
                // file == null && node has 'fileName' property already. this means we're dealing with an update
                // now we need to fill the two 'calculated' properties on the CI if needed.

                // checksum setting on item from node.
                setPropertyFromNode(CHECKSUM_PROPERTY_NAME);
                // fileUri setting on item from node.
                setPropertyFromNode(FILE_URI_PROPERTY_NAME);
            }
        }
    }

    private void setPropertyFromNode(String propertyName) throws RepositoryException {
        if (item.hasProperty(propertyName) && isBlank((String) item.getProperty(propertyName))) {
            item.setProperty(propertyName, node.getProperty(propertyName).getString());
        }
    }

    private ByteSource newInputStreamSupplier(final OverthereFile file) {
        return newInputStreamSupplier(file.getInputStream());
    }

    private ByteSource newInputStreamSupplier(final InputStream is) {
        return new ByteSource() {
            @Override
            public InputStream openStream() throws IOException {
                return is;
            }
        };
    }

    private void writeBinaryData(InputStream dataInput) throws RepositoryException {
        try {
            Binary binary = session.getValueFactory().createBinary(dataInput);
            node.setProperty(DATA_PROPERTY_NAME, binary);
        } finally {
            Closeables.closeQuietly(dataInput);
        }
    }

    private void copyValuesIntoNode() throws RepositoryException {
        Descriptor ciDescriptor = DescriptorRegistry.getDescriptor(item.getType());
        Collection<PropertyDescriptor> propertyDescriptors = Collections2.filter(ciDescriptor.getPropertyDescriptors(), new Predicate<PropertyDescriptor>() {
            @Override
            public boolean apply(PropertyDescriptor input) {
                return !input.isTransient();
            }
        });
        for (PropertyDescriptor pd : propertyDescriptors) {
            if (pd.get(item) == null) {
                removePropertyFromNode(pd);
                continue;
            }

            logger.trace("Writing property [{}] with value [{}]", pd.getFqn(), pd.isPassword() ? "********" : pd.get(item));

            switch (pd.getKind()) {
                case BOOLEAN:
                case INTEGER:
                case STRING:
                case ENUM:
                    copyPrimitivePropertyIntoNode(pd);
                    break;
                case DATE:
                    Calendar cal = new GregorianCalendar(TimeZone.getTimeZone("UTC"));
                    cal.setTime((Date) pd.get(item));
                    node.setProperty(pd.getName(), session.getValueFactory().createValue(cal));
                    break;
                case SET_OF_STRING:
                    copyCollectionOfStringsPropertyIntoNode(pd);
                    break;
                case SET_OF_CI:
                    copyCollectionOfConfigurationItemsPropertyIntoNode(pd);
                    break;
                case LIST_OF_STRING:
                    copyCollectionOfStringsPropertyIntoNode(pd);
                    break;
                case LIST_OF_CI:
                    copyCollectionOfConfigurationItemsPropertyIntoNode(pd);
                    break;
                case CI:
                    copyConfigurationItemPropertyIntoNode(pd);
                    break;
                case MAP_STRING_STRING:
                    copyMapPropertyIntoNode(pd);
                    break;
                default:
                    throw new IllegalArgumentException("Cannot convert property " + pd.getName() + " because it is of unsupported kind " + pd.getKind());

            }
        }
    }

    @SuppressWarnings("unchecked")
    private void copyMapPropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
        MapStringStringView value = MapStringStringView.from((Map<String, String>) pd.get(item));
        if (pd.isPassword()) {
            value = value.encrypt();
        }
        Map<String, String> toWrite = Maps.transformValues(value.getWrapped(), StringValueConverter.valueToString(passwordEncrypter));
        JcrUtils.writeMap(node, pd.getName(), toWrite);
    }

    private void copyMetadata() throws RepositoryException {
        // Made a DateTime so that we can inject different times through tests.
        final DateTime now = DateTime.now(DateTimeZone.UTC);
        setCreatedByMetadata();
        setCreatedAtMetadata(now);
        setLastModifiedByMetadata();
        setLastModifiedAtMetadata(now);

        if (item instanceof BaseConfigurationItem) {
            ((BaseConfigurationItem) item).set$ciAttributes(NodeUtils.readCiAttributes(node));
        }
    }

    private Calendar toCal(DateTime dt) {
        Calendar utc = Calendar.getInstance(TimeZone.getTimeZone("UTC"));
        utc.setTimeInMillis(dt.getMillis());
        return utc;
    }

    private void setLastModifiedAtMetadata(final DateTime now) throws RepositoryException {
        node.setProperty(LAST_MODIFIED_AT_PROPERTY_NAME, toCal(now));
    }

    private void setLastModifiedByMetadata() throws RepositoryException {
        node.setProperty(LAST_MODIFIED_BY_PROPERTY_NAME, Permissions2.getAuthenticatedUserName());
    }

    private void setCreatedByMetadata() throws RepositoryException {
        if (!node.hasProperty(CREATED_BY_PROPERTY_NAME)) {
            String username = Permissions2.getAuthenticatedUserName();
            if (node.hasProperty(LAST_MODIFIED_BY_PROPERTY_NAME)) {
                username = node.getProperty(LAST_MODIFIED_BY_PROPERTY_NAME).getString();
            }
            node.setProperty(CREATED_BY_PROPERTY_NAME, username);
        }
    }

    private void setCreatedAtMetadata(final DateTime now) throws RepositoryException {
        if (!node.hasProperty(CREATED_AT_PROPERTY_NAME)) {
            Calendar when = toCal(now);
            if (node.hasProperty(LAST_MODIFIED_AT_PROPERTY_NAME)) {
                when = node.getProperty(LAST_MODIFIED_AT_PROPERTY_NAME).getDate();
            }
            node.setProperty(CREATED_AT_PROPERTY_NAME, when);
        }
    }

    private void removePropertyFromNode(PropertyDescriptor pd) throws RepositoryException {
        try {
            node.getProperty(pd.getName()).remove();
        } catch (PathNotFoundException ignored) {
        }
    }

    private void copyPrimitivePropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
        String valueAsString = pd.get(item).toString();
        if (pd.getKind() == STRING && pd.isPassword()) {
            valueAsString = passwordEncrypter.ensureEncrypted(valueAsString);
        }
        node.setProperty(pd.getName(), valueAsString);
    }

    @SuppressWarnings("unchecked")
    private void copyCollectionOfStringsPropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
        final ValueFactory valueFactory = session.getValueFactory();
        final StringValueConverter converter = new StringValueConverter(passwordEncrypter);

        Collection<?> valueAsCollection = convertObjectValueToCollection(pd, pd.get(item));
        List<Value> jcrValueList = newArrayList();
        if (!(valueAsCollection instanceof AbstractStringView)) {
            valueAsCollection = valueAsCollection instanceof Set ? SetOfStringView.from((Set<String>) valueAsCollection) : ListOfStringView.from((List<String>) valueAsCollection);
        }

        @SuppressWarnings("rawtypes")
        AbstractStringView view = (AbstractStringView) valueAsCollection;
        if (pd.isPassword()) {
            view = view.encrypt();
        }
        for (Object o : view.getWrapped()) {
            if (!(o instanceof StringValue)) {
                throw new IllegalArgumentException("Element in property " + pd.getName() + " of configuration item " + item.getId()
                        + " is not a StringValue: " + o);
            }
            StringValue stringValue = (StringValue) o;
            jcrValueList.add(valueFactory.createValue(converter.convert(stringValue)));
        }

        Value[] jcrValues = jcrValueList.toArray(new Value[jcrValueList.size()]);

        // if property is single-value we need to remove it so we can create a new multi-value property
        if (node.hasProperty(pd.getName())) {
            Property property = node.getProperty(pd.getName());
            if(!property.isMultiple() || jcrValues.length == 0){
                property.remove();
            }
        }

        if (jcrValues.length > 0) {
            node.setProperty(pd.getName(), jcrValues);
        }
    }

    private void copyConfigurationItemPropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
        if (pd.isAsContainment()) {
            return;
        }

        String referencedCiId = ((ConfigurationItem) pd.get(item)).getId();
        Node referencedCi = session.getNode(getAbsolutePathFromId(referencedCiId));
        node.setProperty(pd.getName(), referencedCi);
    }

    private void copyCollectionOfConfigurationItemsPropertyIntoNode(PropertyDescriptor pd) throws RepositoryException {
        if (pd.isAsContainment() && pd.getKind() == SET_OF_CI) {
            return;
        }

        Collection<?> valueAsCollection = convertObjectValueToCollection(pd, pd.get(item));
        Set<String> set = new LinkedHashSet<>();
        List<Value> jcrReferenceList = newArrayList();

        for (Object each : valueAsCollection) {
            if (!(each instanceof ConfigurationItem)) {
                throw new IllegalArgumentException("Element in property " + pd.getName() + " of repository entity " + item.getId()
                        + " is not a ConfigurationItem: " + each);
            }
            String referencedCiId = ((ConfigurationItem) each).getId();
            set.add(referencedCiId);
            Node referencedCi = session.getNode(getAbsolutePathFromId(referencedCiId));
            if (pd.isAsContainment() && !referencedCi.getParent().getPath().equals(getAbsolutePathFromId(item.getId()))) {
                throw new Checks.IncorrectArgumentException("Cannot add [%s] to property [%s] of item [%s], as it is not contained by it.", referencedCiId, pd.getName(), item.getId());
            }
            jcrReferenceList.add(session.getValueFactory().createValue(referencedCi));
        }

        if (pd.isAsContainment() && pd.getKind() == LIST_OF_CI) {
            Map<String, Node> referencesFromProperty = getReferencesFromProperty(node, pd.getName());
            for (String s : referencesFromProperty.keySet()) {
                if (set.add(getIdFromAbsolutePath(s))) {
                    jcrReferenceList.add(session.getValueFactory().createValue(referencesFromProperty.get(s)));
                }
            }
        }

        Value[] jcrReferenceValues = jcrReferenceList.toArray(new Value[jcrReferenceList.size()]);

//        if (node.hasProperty(pd.getName())) {
//            node.getProperty(pd.getName()).remove();
//        }

//        if (jcrReferenceValues.length > 0) {
            node.setProperty(pd.getName(), jcrReferenceValues);
//        }
    }

    private Collection<?> convertObjectValueToCollection(PropertyDescriptor pd, Object value) {
        if (!(value instanceof Collection)) {
            throw new IllegalArgumentException("Property " + pd.getName() + " of repository entity " + item.getId() + " is not a Collection: " + value);
        }
        return (Collection<?>) value;
    }

    private void updateParentForListAsContainmentRelation() throws RepositoryException {
        Node parent = node.getParent();
        if (parent.getPath().equals(session.getRootNode().getPath())) {
            return;
        }

        Type parentType = typeOf(parent);
        PropertyDescriptor pd = findContainmentListProperty(parentType, item.getType());
        if (pd == null) {
            return;
        }

        String referencingPropertyName = pd.getName();
        Map<String, Node> nodes = getReferencesFromProperty(parent, referencingPropertyName);
        nodes.put(node.getPath(), node);

        List<Value> values = newArrayList();
        for (Node n : nodes.values()) {
            values.add(session.getValueFactory().createValue(n));
        }
        parent.setProperty(referencingPropertyName, values.toArray(new Value[values.size()]));
    }

    private Map<String, Node> getReferencesFromProperty(Node node, String propertyName) throws RepositoryException {
        Map<String, Node> nodes = newLinkedHashMap();
        if (node.hasProperty(propertyName)) {
            Property referencingProperty = node.getProperty(propertyName);
            for (Value value : referencingProperty.getValues()) {
                Node referencedCiNode = getReferencedCiNode(node, value, session);
                nodes.put(referencedCiNode.getPath(), referencedCiNode);
            }
        }
        return nodes;
    }


    private static final Logger logger = LoggerFactory.getLogger(NodeWriter.class);
}
