package com.xebialabs.xlplatform.repository.jcrsql2

import com.xebialabs.xlplatform.repository.jcrsql2.Operator._

sealed trait Operator {
  def build(): String = this match {
    case Equals => "="
    case NotEquals => "!="
    case LessThan => "<"
    case LessThanEquals => "<="
    case GreaterThan => ">"
    case GreaterThanEquals => ">="
    case Like => "LIKE"
    case NotLike => "NOT LIKE"
  }
}

object Operator {
  case object Equals extends Operator
  case object NotEquals extends Operator
  case object LessThan extends Operator
  case object LessThanEquals extends Operator
  case object GreaterThan extends Operator
  case object GreaterThanEquals extends Operator
  case object Like extends Operator
  case object NotLike extends Operator
}

sealed trait DynamicOperand {
  def build(): String = this match {
    case PropertyValue(selector, property) => s"${selector.name}.[$property]"
    case ReferenceValue(pv) => s"REFERENCE(${pv.build()})"
    case Length(pv) => s"LENGTH(${pv.build()}"
    case Name(selector) => s"NAME(${selector.name})"
    case LocalName(selector) => s"LOCALNAME(${selector.name})"
    case Depth(selector) => s"DEPTH(${selector.name})"
    case Path(selector) => s"PATH(${selector.name})"
    case ChildCount(selector) => s"CHILDCOUNT(${selector.name})"
    case FullTextSearchScore(selector) => s"SCORE(${selector.name})"
    case Upper(dynamicOperand) => s"UPPER(${dynamicOperand.build()}"
    case Lower(dynamicOperand) => s"LOWER(${dynamicOperand.build()}"
  }
}
case class PropertyValue(selector: Selector, property: String) extends DynamicOperand
case class ReferenceValue(pv: PropertyValue) extends DynamicOperand
case class Length(pv: PropertyValue) extends DynamicOperand
case class Name(selector: Selector) extends DynamicOperand
case class LocalName(selector: Selector) extends DynamicOperand
case class Depth(selector: Selector) extends DynamicOperand
case class Path(selector: Selector) extends DynamicOperand
case class ChildCount(selector: Selector) extends DynamicOperand
case class FullTextSearchScore(selector: Selector) extends DynamicOperand
case class Upper(dynOp: DynamicOperand) extends DynamicOperand
case class Lower(dynOp: DynamicOperand) extends DynamicOperand



sealed trait Constraint {
  def build(params: QueryParameters): String
  def and(constraint: Constraint): Constraint = ConstraintGroup(constraint :: And :: this :: Nil)
  def or(constraint: Constraint): Constraint = ConstraintGroup(constraint :: Or :: this :: Nil)
  def not = Not(this)
}

case class ConstraintGroup(constraints: List[Constraint]) extends Constraint {
  override def build(params: QueryParameters): String = s"(${constraints.reverse.mkString(" ")})"
  override def and(constraint: Constraint): Constraint = copy(constraint :: And :: constraints)
  override def or(constraint: Constraint): Constraint = copy(constraint :: Or :: constraints)
}

sealed trait ConstraintItem extends Constraint

case object EmptyConstraint extends Constraint {
  def build(params: QueryParameters): String = ""
  override def and(constraint: Constraint): Constraint = constraint
  override def or(constraint: Constraint): Constraint = constraint
}

case object And extends ConstraintItem {
  override def build(params: QueryParameters): String = "AND"
}
case object Or extends ConstraintItem {
  override def build(params: QueryParameters): String = "OR"
}

case class Not(constraint: Constraint) extends ConstraintItem {
  override def build(params: QueryParameters): String = constraint match {
    case group: ConstraintGroup => s"NOT ${constraint.build(params)}"
    case _ => s"NOT (${constraint.build(params)})"
  }
}

case class Comparison(dynamicOperand: DynamicOperand, operator: Operator, value: AnyRef) extends ConstraintItem {
  override def build(params: QueryParameters): String = s"${dynamicOperand.build()} ${operator.build()} \$${params.bind(value)}"
}

case class PropertyExistence(pv: PropertyValue) extends ConstraintItem {
  override def build(params: QueryParameters): String = s"${pv.build()} IS NOT NULL"
}

case class ChildNode(selector: Selector, value: String) extends ConstraintItem {
  override def build(params: QueryParameters): String = s"ISCHILDNODE(${selector.name}, \$${params.bind(value)})"
}

case class DescendantNode(selector: Selector, value: String) extends ConstraintItem {
  override def build(params: QueryParameters): String = s"ISDESCENDANTNODE(${selector.name}, \$${params.bind(value)})"
}

case class FullTextSearch(pv: PropertyValue, value: String) extends ConstraintItem {
  override def build(params: QueryParameters): String = s"CONTAINS(${pv.build()}, \$${params.bind(value)})"
}

