package com.xebialabs.deployit.jcr;

import com.google.common.annotations.VisibleForTesting;
import com.xebialabs.deployit.repository.QueryTemplate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.*;
import javax.jcr.query.Query;
import javax.jcr.query.QueryManager;
import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Map;

import static com.google.common.collect.Maps.newLinkedHashMap;
import static javax.jcr.query.Query.JCR_SQL2;

/**
 * Helper class to create JCR queries from a string. Contains query string (JCR_SQL2), parameters and pagination metadata. You can create {@link javax.jcr.query.Query} objects using {@link #createQuery(javax.jcr.Session)}.
 */
public class JcrQueryTemplate implements QueryTemplate {

    private final String queryString;
    private final Map<String, Object> parameters = newLinkedHashMap();

    private long page = 0;
    private long resultsPerPage = 0;
    private int depth = Integer.MAX_VALUE;

    /**
     * Creates a new query template
     *
     * @param query a JCR SQL2 query.
     */
    public JcrQueryTemplate(String query) {
        this.queryString = query;
    }

    /**
     * Creates a new query template
     *
     * @param query a JCR SQL2 query.
     * @param parameters map of parameters mentioned in the query.
     */
    public JcrQueryTemplate(String query, Map<String, Object> parameters) {
        this.queryString = query;
        if (parameters != null && !parameters.isEmpty()) {
            this.parameters.putAll(parameters);
        }
    }

    //
    // Set query parameters
    //

    /**
     * Sets the value of a parameter.
     *
     * @param key the parameter as mentioned in the query string.
     * @param value the value of the parameter.
     * @return this
     */
    public QueryTemplate setParameter(String key, Object value) {
        parameters.put(key, value);
        return this;
    }

    public QueryTemplate setPage(long page) {
        this.page = page;
        return this;
    }

    public QueryTemplate setResultsPerPage(long resultsPerPage) {
        this.resultsPerPage = resultsPerPage;
        return this;
    }

    public QueryTemplate setDepth(int depth) {
        this.depth = depth;
        return this;
    }

    @VisibleForTesting
    String getQueryString() {
        return queryString;
    }
//
    // Create JCR query
    //

    /**
     * Creates a new JCR Query object from the template.
     *
     * @param session the JCR session
     * @return a new query.
     * @throws javax.jcr.RepositoryException
     */
    public Query createQuery(final Session session) throws RepositoryException {

        QueryManager qm = session.getWorkspace().getQueryManager();
        Query query = qm.createQuery(queryString, JCR_SQL2);

        addBindings(session.getValueFactory(), query);
        addPagingInfo(query);

        logger.trace("JCR query built: {}", query.getStatement());

        return query;
    }

    private void addBindings(final ValueFactory valueFactory, final Query query) throws RepositoryException {
        for (Map.Entry<String, Object> parameter: parameters.entrySet()) {
            query.bindValue(parameter.getKey(), toValue(valueFactory, parameter.getValue()));
        }
    }

    @VisibleForTesting
    void addPagingInfo(final Query query) {
        if (resultsPerPage > 0) {
            query.setLimit(resultsPerPage);
            query.setOffset(page * resultsPerPage);
        }
    }

    private static Value toValue(ValueFactory factory, Object valueObject) throws RepositoryException {
        if (valueObject instanceof String) {
            return factory.createValue((String) valueObject);
        } else if (valueObject instanceof Calendar) {
            return factory.createValue((Calendar) valueObject);
        } else if (valueObject instanceof BigDecimal) {
            return factory.createValue((BigDecimal) valueObject);
        } else if (valueObject instanceof Binary) {
            return factory.createValue((Binary) valueObject);
        } else if (valueObject instanceof Boolean) {
            return factory.createValue((Boolean) valueObject);
        } else if (valueObject instanceof Double) {
            return factory.createValue((Double) valueObject);
        } else if (valueObject instanceof Long) {
            return factory.createValue((Long) valueObject);
        } else if (valueObject instanceof Node) {
            return factory.createValue((Node) valueObject);
        } else {
            throw new IllegalArgumentException("Unsupported value type for JCR query: " + valueObject.getClass().getName());
        }
    }

    @Override
    public String toString() {
        return "JcrQueryTemplate [queryString=" + queryString + ", parameters=" + parameters + ", page=" + page + ", resultsPerPage=" + resultsPerPage
            + ", depth = " + depth + "]";
    }

    public int getDepth() {
        return depth;
    }

    private static final Logger logger = LoggerFactory.getLogger(JcrQueryTemplate.class);
}
