package com.xebialabs.deployit.repository;

import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.engine.spi.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.util.PasswordEncrypter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.nodetype.NodeType;
import javax.jcr.version.Version;
import javax.jcr.version.VersionException;
import javax.jcr.version.VersionHistory;
import javax.jcr.version.VersionIterator;
import java.util.Calendar;
import java.util.List;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.jcr.JcrConstants.ARTIFACT_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.LAST_MODIFIED_AT_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.LAST_MODIFIED_BY_PROPERTY_NAME;
import static com.xebialabs.deployit.repository.JcrPathHelper.getAbsolutePathFromId;

@Component
public class JcrHistoryService implements HistoryService {

    private final JcrTemplate jcrTemplate;
    private final PasswordEncrypter passwordEncrypter;

    @Autowired
    public JcrHistoryService(JcrTemplate jcrTemplate, PasswordEncrypter passwordEncrypter) {
        this.jcrTemplate = jcrTemplate;
        this.passwordEncrypter = passwordEncrypter;
    }

    @Override
    public List<ConfigurationItemRevision> getVersionRevisions(final String id) {
        logger.debug("Retrieving version history for node {}.", id);

        return jcrTemplate.execute(new JcrCallback<List<ConfigurationItemRevision>>() {
            @Override
            public List<ConfigurationItemRevision> doInJcr(final Session session) throws RepositoryException {
                final List<ConfigurationItemRevision> items = newArrayList();
                String absPath = getAbsolutePathFromId(id);
                if (session.getNode(absPath).isNodeType(NodeType.MIX_VERSIONABLE)) {
                    final VersionHistory history = session.getWorkspace().getVersionManager().getVersionHistory(absPath);
                    Version rootVersion = history.getRootVersion();
                    final VersionIterator versions = history.getAllLinearVersions();
                    while (versions.hasNext()) {
                        Version version = versions.nextVersion();
                        if (version.getName().equals(rootVersion.getName())) {
                            continue;
                        }
                        String name = version.getName();
                        Node frozenNode = version.getFrozenNode();
                        items.add(getConfigurationItemRevision(name, frozenNode));
                    }
                }
                items.add(getConfigurationItemRevision(CURRENT_REVISION, session.getNode(absPath)));

                return items;
            }
        });
    }

    private ConfigurationItemRevision getConfigurationItemRevision(final String name, final Node frozenNode) throws RepositoryException {
        Calendar created = frozenNode.getProperty(LAST_MODIFIED_AT_PROPERTY_NAME).getDate();
        String username = "<system>";
        if (frozenNode.hasProperty(LAST_MODIFIED_BY_PROPERTY_NAME)) {
            username = frozenNode.getProperty(LAST_MODIFIED_BY_PROPERTY_NAME).getString();
        }
        return new ConfigurationItemRevision(name, created, username);
    }

    @Override
    public <T extends ConfigurationItem> T readRevision(final String id, final String revisionName) {
        logger.debug("Retrieving revision [{}] for [{}]", revisionName, id);
        return jcrTemplate.execute(new JcrCallback<T>() {
            @Override
            public T doInJcr(final Session session) throws RepositoryException {
                final VersionHistory history = session.getWorkspace().getVersionManager().getVersionHistory(getAbsolutePathFromId(id));
                Version version = getVersionByRevisionName(history, revisionName, id);
                return readNode(session, version, id);
            }
        });
    }

    public <T extends ConfigurationItem> T readRevisionByLabel(final String id, final String label) {
        logger.debug("Retrieving revision labelled as [{}] for  label[{}]", label, id);
        return jcrTemplate.execute(new JcrCallback<T>() {
            @Override
            public T doInJcr(final Session session) throws RepositoryException {
                final VersionHistory history = session.getWorkspace().getVersionManager().getVersionHistory(getAbsolutePathFromId(id));
                Version version;
                try {
                    version = history.getVersionByLabel(label);
                } catch (VersionException ve) {
                    throw new UnknownLabelException("Cannot find revision labelled as [%s] for configuration item [%s]", label, id);
                }
                return readNode(session, version, id);
            }
        });
    }

    public void labelRevision(final String id, final String revisionName, final String label) {
        logger.debug("Labelling revision [{}] for [{}] with label [{}]", revisionName, id, label);
        jcrTemplate.execute(new JcrCallback<Void>() {
            @Override
            public Void doInJcr(final Session session) throws RepositoryException {
                final VersionHistory history = session.getWorkspace().getVersionManager().getVersionHistory(getAbsolutePathFromId(id));
                Version version = getVersionByRevisionName(history, revisionName, id);
                history.addVersionLabel(version.getName(), label, true);

                return null;
            }
        });
    }

    private Version getVersionByRevisionName(VersionHistory history, String revisionName, String id) throws RepositoryException {
        Version version;
        try {
            version = history.getVersion(revisionName);
        } catch (VersionException ve) {
            throw new UnknownRevisionException("Cannot find revision [%s] for configuration item [%s]", revisionName, id);
        }
        return version;
    }

    private <T extends ConfigurationItem> T readNode(Session session, Version version, String id) throws RepositoryException {
        Node node = version.getFrozenNode();
        // Your IDE might not think <T> is needed here, but your compiler does ;)
        T item = NodeReader.read(session, node, Integer.MAX_VALUE, null, passwordEncrypter);
        // The ID set by the XLrNodeReader will be the path of the frozen node minus the leading slash. Overwrite that with the ID the user expects.
        item.setId(id);
        return item;
    }

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 404)
    public static class UnknownRevisionException extends DeployitException {
        public UnknownRevisionException(final String messageTemplate, final Object... params) {
            super(messageTemplate, params);
        }
    }

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 404)
    public static class UnknownLabelException extends DeployitException {
        public UnknownLabelException(final String messageTemplate, final Object... params) {
            super(messageTemplate, params);
        }
    }


    private static final Logger logger = LoggerFactory.getLogger(JcrHistoryService.class);
}
