package com.xebialabs.deployit.repository;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.engine.spi.event.*;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.jcr.JcrQueryTemplate;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.license.service.LicenseService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.query.Query;
import javax.jcr.query.QueryResult;
import javax.jcr.query.RowIterator;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.repository.JcrPathHelper.getAbsolutePathFromId;
import static com.xebialabs.deployit.repository.SearchQueryBuilder.CI_SELECTOR_NAME;

public class JcrRepositoryService implements RepositoryService {

    private final JcrTemplate jcrTemplate;
    private final PasswordEncrypter passwordEncrypter;

    private LicenseService licenseService;

    @Autowired
    public JcrRepositoryService(JcrTemplate jcrTemplate, PasswordEncrypter passwordEncrypter, LicenseService licenseService) {
        this.jcrTemplate = jcrTemplate;
        this.passwordEncrypter = passwordEncrypter;
        this.licenseService = licenseService;
    }

    @PostConstruct
    public void initializeLicenseValidator() {
        licenseService.initialize(this);
    }

    @Override
    public boolean exists(final String id) {
        logger.debug("Checking whether node [{}] exists.", id);

        return jcrTemplate.execute(new JcrCallback<Boolean>() {
            @Override
            public Boolean doInJcr(final Session session) throws RepositoryException {
                return session.itemExists(getAbsolutePathFromId(id));
            }
        });
    }

    @Override
    public <T extends ConfigurationItem> T read(final String id) {
        return read(id, Integer.MAX_VALUE, null, true);
    }

    @Override
    public <T extends ConfigurationItem> T read(final String id, int depth) {
        return read(id, depth, null, true);
    }

    @Override
    public <T extends ConfigurationItem> T read(String id, WorkDir workDir) {
        return read(id, Integer.MAX_VALUE, workDir, true);
    }

    @Override
    public <T extends ConfigurationItem> T read(final String id, final boolean useCache) {
        return read(id, Integer.MAX_VALUE, null, useCache);
    }

    @Override
    public <T extends ConfigurationItem> T read(final String id, final WorkDir workDir, final boolean useCache) {
        return read(id, Integer.MAX_VALUE, workDir, useCache);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T extends ConfigurationItem> T read(final String id, final int depth, final WorkDir workDir, final boolean useCache) {
        logger.debug("Reading node [{}] with depth {} and workdir {} (useCache={})", id, depth, workDir, useCache);
        checkNotNull(id, "id is null");

        return (T) _read(id, depth, workDir, useCache, true);
    }

    private <T extends ConfigurationItem> T _read(final String id, final int depth, final WorkDir workDir, final boolean useCache, final boolean failOnMissing) {
        return jcrTemplate.execute(new JcrCallback<T>() {
            @Override
            public T doInJcr(final Session session) throws RepositoryException {
                String absolutePathFromId = getAbsolutePathFromId(id);
                boolean exists = session.nodeExists(absolutePathFromId);
                if (failOnMissing && !exists) {
                    throw new NotFoundException("Repository entity [%s] not found", id);
                } else if (!exists) {
                    return null;
                }

                Node node = session.getNode(absolutePathFromId);
                if (useCache) {
                    return NodeReader.<T>read(session, node, depth, workDir, passwordEncrypter);
                } else {
                    return NodeReader.<T>read(session, node, depth, workDir, new NodeReaderContext(), passwordEncrypter);
                }
            }
        });
    }

    @Override
    public List<ConfigurationItemData> list(SearchParameters parameters) {
        checkNotNull(parameters, "parameters is null");
        logger.debug("Listing node IDs with parameters {}.", parameters);

        QueryTemplate queryTemplate = new SearchQueryBuilder(parameters).createTemplate();
        return list(queryTemplate);
    }

    @Override
    public <T extends ConfigurationItem> List<T> listEntities(SearchParameters parameters) {
        checkNotNull(parameters, "parameters is null");
        logger.debug("Listing nodes with parameters {}.", parameters);

        QueryTemplate queryTemplate = new SearchQueryBuilder(parameters).createTemplate();
        return listEntities(queryTemplate);
    }

    @Override
    public List<ConfigurationItemData> list(QueryTemplate queryTemplate) {
        checkNotNull(queryTemplate, "queryTemplate is null");
        logger.debug("Listing node IDs with JCR query template {}.", queryTemplate);

        // Setting explicit depth=0 as no depth is need
        queryTemplate.setDepth(0);

        List<ConfigurationItem> entities = listEntities(queryTemplate);
        return Lists.transform(entities, new Function<ConfigurationItem, ConfigurationItemData>() {
            public ConfigurationItemData apply(ConfigurationItem input) {
                return new ConfigurationItemData(input.getId(), input.getType());
            }
        });
    }

    @Override
    public <T extends ConfigurationItem> List<T> listEntities(final QueryTemplate queryTemplate) {
        checkNotNull(queryTemplate, "query is null");
        logger.debug("Listing nodes with JCR query template {}.", queryTemplate);

        return jcrTemplate.execute(new JcrCallback<List<T>>() {
            @Override
            public List<T> doInJcr(final Session session) throws RepositoryException {
                final Query query = ((JcrQueryTemplate) queryTemplate).createQuery(session);
                final QueryResult queryResult = query.execute();
                return getOrderedEntities(session, queryResult, queryTemplate.getDepth());
            }
        });
    }

    private <T extends ConfigurationItem> List<T> getOrderedEntities(Session session, QueryResult queryResult, int depth) throws RepositoryException {
        final List<T> items = newArrayList();
        final RowIterator iterator = queryResult.getRows();
        while (iterator.hasNext()) {
            Node node = iterator.nextRow().getNode(CI_SELECTOR_NAME);
            // Your IDE might not think <T> is needed here, but your compiler does ;)
            T item = NodeReader.<T>read(session, node, depth, null, passwordEncrypter);
            items.add(item);
        }
        return items;
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T extends ConfigurationItem> void create(final T... cis) {
        ChangeSet changeset = new ChangeSet();
        changeset.create(newArrayList(cis));
        execute(changeset);

        EventBusHolder.publish(new CisCreatedEvent(cis));
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T extends ConfigurationItem> void update(final T... cis) {
        ChangeSet changeset = new ChangeSet();
        changeset.update(newArrayList(cis));
        execute(changeset);

        EventBusHolder.publish(new CisUpdatedEvent(changeset.getUpdateCis()));
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T extends ConfigurationItem> void createOrUpdate(T... cis) {
        ChangeSet changeSet = new ChangeSet();
        changeSet.createOrUpdate(newArrayList(cis));
        execute(changeSet);

        EventBusHolder.publish(new CisCreatedEvent(newArrayList(changeSet.getCreateOrUpdateActualCreatedCis())));
        EventBusHolder.publish(new CisUpdatedEvent(newArrayList(changeSet.getCreateOrUpdateActualUpdatedCis())));
    }

    @Override
    public void delete(final String... ids) {
        ChangeSet changeset = new ChangeSet();
        // Read for auditing
        List<ConfigurationItem> cis = new ArrayList<>();
        for (String id : ids) {
            ConfigurationItem e = _read(id, 0, null, true, false);
            if (e != null) {
                cis.add(e);
            }
        }

        changeset.delete(newArrayList(ids));
        checkReferentialIntegrity(changeset);
        execute(changeset);


        EventBusHolder.publish(new CisDeletedEvent(cis));
    }

    @Override
    public void rename(final String id, final String newName) {
        checkArgument(newName.indexOf('/') == -1, "New name [%s] should not contain a /", newName);
        ConfigurationItem configurationItem = _read(id, 0, null, true, true);
        ChangeSet changeSet = new ChangeSet();
        changeSet.rename(id, newName);
        execute(changeSet);

        EventBusHolder.publish(new CiRenamedEvent(configurationItem, newName));
    }

    @Override
    public <T extends ConfigurationItem> void move(final String id, final String newId) {
        logger.debug("Moving node [{}] to [{}]", id, newId);

        ConfigurationItem configurationItem = _read(id, 0, null, true, true);
        ChangeSet changeSet = new ChangeSet();
        changeSet.move(id, newId);
        execute(changeSet);


        EventBusHolder.publish(new CiMovedEvent(configurationItem, newId));
    }

    @Override
    public void copy(final String id, final String newId) {
        ChangeSet changeSet = new ChangeSet();
        changeSet.copy(id, newId);
        execute(changeSet);
        ConfigurationItem configurationItem = _read(id, 0, null, true, true);

        EventBusHolder.publish(new CiCopiedEvent(configurationItem, newId));
    }

    /**
     * Checks if a type may be moved. Defaults is no-op. Subclasses may override this method and throw a
     * {@link RuntimeException} if they want to prevent the type from being moved.
     *
     * @param type
     *            the type that is to be moved
     * @throws RuntimeException
     *             if the move is not allowed on the type.
     */
    protected void checkMoveAllowed(Type type) {
    }

    /**
     * Checks if a type may be copied. Defaults is no-op. Subclasses may override this method and throw a
     * {@link RuntimeException} or {@link javax.jcr.RepositoryException} if they want to prevent the type from being moved.
     *
     * @param session a JCR session so you can access the repository to do the checking
     * @param toBeCopied the path of the CI that will be copied
     * @param newId the destination path of the CI
     *
     * @throws javax.jcr.RepositoryException
     *             if the copy is not allowed on the type.
     * @throws RuntimeException
     *             if the copy is not allowed on the type.
     */
    protected void checkCopyAllowed(Session session, String toBeCopied, String newId) throws RepositoryException {
    }


    static Type readType(Session session, String id) throws RepositoryException {
        return Type.valueOf(session.getNode(getAbsolutePathFromId(id)).getProperty(JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME).getString());
    }

    static String extractParentId(String id) {
        int indexOf = id.lastIndexOf('/');
        if (indexOf != -1) {
            return id.substring(0,indexOf);
        }
        return id;
    }

    @Override
    public void execute(ChangeSet changeset) {
        jcrTemplate.execute(new JcrChangeSetExecutor(this, changeset, passwordEncrypter, licenseService));
    }

    @Override
    public void checkReferentialIntegrity(final ChangeSet changeset) throws ItemInUseException, ItemAlreadyExistsException {
        jcrTemplate.execute(new JcrReferentialIntegrityChecker(this, changeset, passwordEncrypter, licenseService));
    }

    @Override
    public QueryTemplateFactory getQueryTemplateFactory() {
        return new QueryTemplateFactory() {
            @Override
            public QueryTemplate createTemplate(String query) {
                return new JcrQueryTemplate(query);
            }

            @Override
            public QueryTemplate createTemplate(String query, Map<String, Object> parameters) {
                return new JcrQueryTemplate(query, parameters);
            }
        };
    }

    private static Logger logger = LoggerFactory.getLogger(JcrRepositoryService.class);

}
