package com.xebialabs.xlplatform.repository

import javax.jcr.{Node, Property, Session}

import com.google.common.base.{Preconditions, Strings}
import com.xebialabs.deployit.jcr.{JcrCallback, JcrConstants, JcrTemplate}
import com.xebialabs.deployit.repository.RepositoryMetadataService
import com.xebialabs.license.LicenseRepositoryIdException
import grizzled.slf4j.Logging

class JcrRepositoryMetadataService(jcrTemplate: JcrTemplate) extends RepositoryMetadataService with Logging {

  // Lazy initialised to prevent frequent querying
  private var storedRepoId: Option[String] = None

  /**
   * Writes the repositoryId. This is a write once value.
   * @param repositoryId unique id for this repository
   * @throws LicenseRepositoryIdException if the repository has already an id
   */

  override def validateAndStoreRepositoryId(repositoryId: String): Unit = {
    Preconditions.checkArgument(!Strings.isNullOrEmpty(repositoryId))
    getRepositoryId match {
      case Some(current) => if (current != repositoryId) throw new LicenseRepositoryIdException(current, repositoryId)
      case None => saveRepositoryId(repositoryId)
    }
  }

  def saveRepositoryId(repositoryId: String): Unit =
    jcrTemplate.execute(new JcrCallback[String]() {
      def doInJcr(session: Session): String = {
        val configurationNode: Node = session.getNode(JcrConstants.CONFIGURATION_NODE_ID)
        val property: Property = configurationNode.setProperty(JcrConstants.REPOSITORY_ID_PROPERTY_NAME, repositoryId)
        session.save()
        property.getString
      }
    })

  private def getRepositoryId: Option[String] = {
    storedRepoId match {
      case Some(_) =>
      case None => storedRepoId = queryRepositoryId
    }
    storedRepoId
  }

  private def queryRepositoryId: Option[String] = {
    jcrTemplate.execute(new JcrCallback[Option[String]]() {
      def doInJcr(session: Session): Option[String] = {
        val configurationNode = session.getNode(JcrConstants.CONFIGURATION_NODE_ID)
        if (configurationNode.hasProperty(JcrConstants.REPOSITORY_ID_PROPERTY_NAME)) {
          Some(configurationNode.getProperty(JcrConstants.REPOSITORY_ID_PROPERTY_NAME).getString)
        }
        else {
          None
        }
      }
    })
  }
}
