package com.xebialabs.xlplatform.repository

import java.util.Calendar
import javax.jcr._
import javax.jcr.nodetype.{NodeType, NodeTypeManager, NodeTypeTemplate}

import com.xebialabs.deployit.jcr.JcrConstants._
import com.xebialabs.deployit.jcr.JcrTemplate
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot
import com.xebialabs.deployit.repository.internal.Root
import grizzled.slf4j.Logging

import scala.util.Try

class XlRepositoryInitializer(repository: Repository, config: XlRepositoryConfig) extends Logging {
  def init(): Unit = {
    val creds: Credentials = config.credentials.orNull
    val session: Session = repository.login(creds)
    try {
      if (isInitialized(session)) {
        logger.info("Nodetypes already initialized!")
      } else {
        initializeNodeTypes(session)
      }
      doInitialize(session)
    } finally {
      session.logout()
    }
  }

  private[this] def initializeNodeTypes(implicit session: Session): Unit = {
    session.getWorkspace.getNamespaceRegistry.registerNamespace(DEPLOYIT_NAMESPACE_PREFIX, DEPLOYIT_NAMESPACE_URI)

    Seq(CONFIGURATION_ITEM_NODETYPE_NAME,
      ARTIFACT_NODETYPE_NAME,
      TASK_NODETYPE_NAME,
      STEP_NODETYPE_NAME,
      CONFIGURATION_NODETYPE_NAME).foreach(createMixinType)
  }

  private[this] def doInitialize(implicit session: Session): Unit = {

    ConfigurationItemRoot.values().filter(_ != ConfigurationItemRoot.NESTED).map(_.getRootNodeName).foreach(createXlRoot)

    if (!existsNode(CONFIGURATION_NODE_NAME)) {
      createNode(CONFIGURATION_NODE_NAME).addMixin(CONFIGURATION_NODETYPE_NAME)
    }

    Seq(TASKS_NODE_NAME, SECURITY_NODE_NAME, ROLES_NODE_NAME, ROLE_ASSIGNMENTS_NODE_NAME).foreach(createNodeIfNotExists)

    storeRepoVersion
    storeRepositoryId

    session.save()
  }

  private[this] def isInitialized(session: Session): Boolean = {
    Try(session.getNamespaceURI(DEPLOYIT_NAMESPACE_PREFIX))
      .map(_ => true)
      .recover({case nse: NamespaceException => false})
      .get
  }

  private[this] def createMixinType(typeName: String)(implicit session: Session) = {
    val manager: NodeTypeManager = session.getWorkspace.getNodeTypeManager
    val nodeTypeTemplate: NodeTypeTemplate = manager.createNodeTypeTemplate
    nodeTypeTemplate.setName(typeName)
    nodeTypeTemplate.setQueryable(true)
    nodeTypeTemplate.setAbstract(false)
    nodeTypeTemplate.setMixin(true)
    val nodeType: NodeType = manager.registerNodeType(nodeTypeTemplate, false)
    logger.debug(s"Registered NodeType definition: $nodeType")
  }

  private[this] def createXlRoot(rootName: String)(implicit session: Session) = {
    if (!existsNode(rootName)) {
      val node: Node = createNode(rootName)
      node.addMixin(CONFIGURATION_ITEM_NODETYPE_NAME)
      node.addMixin(NodeType.MIX_VERSIONABLE)
      node.setProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME, "internal." + classOf[Root].getSimpleName)
      node.setProperty(LAST_MODIFIED_AT_PROPERTY_NAME, Calendar.getInstance)
      logger.debug(s"Configured XL Rootnode $rootName at $node")
    }
  }

  private[this] def existsNode(nodeName: String)(implicit session: Session) = {
    session.getRootNode.hasNode(nodeName)
  }

  private[this] def createNodeIfNotExists(nodeName: String)(implicit session: Session) = {
    if (!existsNode(nodeName)) {
      createNode(nodeName)
    }
  }

  private[this] def createNode(nodeName: String)(implicit session: Session) = {
    val node: Node = session.getRootNode.addNode(nodeName)
    logger.debug(s"Created node $node")
    node
  }

  private[this] def storeRepoVersion(implicit session: Session): Unit = {
    if (!existsNode(VERSIONS_NODE_NAME)) {
      val node: Node = session.getRootNode.addNode(VERSIONS_NODE_NAME)
      logger.debug(s"Initialized repository version node at $node")
    }
  }

  private[this] def storeRepositoryId(implicit session: Session) = {
    val template = new JcrTemplate(repository, config.credentials.orNull)
    val metadataService = new JcrRepositoryMetadataService(template)

    config.repositoryId.foreach { id =>
        metadataService.validateAndStoreRepositoryId(id)
    }
  }
}