package com.xebialabs.deployit.repository;

import com.xebialabs.deployit.booter.local.CiRoots;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.jcr.RuntimeRepositoryException;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.repository.core.Directory;
import com.xebialabs.deployit.repository.internal.Root;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.deployit.util.Tuple;
import com.xebialabs.license.service.LicenseService;
import com.xebialabs.license.service.LicenseTransaction;
import com.xebialabs.xlplatform.artifact.resolution.ArtifactResolverRegistry;
import com.xebialabs.xlplatform.utils.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import scala.collection.Seq;

import javax.jcr.*;
import javax.jcr.nodetype.NodeType;
import javax.jcr.version.Version;
import javax.jcr.version.VersionHistory;
import javax.jcr.version.VersionIterator;
import javax.jcr.version.VersionManager;
import java.util.*;
import java.util.function.Predicate;
import java.util.stream.Stream;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.deployit.io.ArtifactFileUtils.hasRealOrResolvedFile;
import static com.xebialabs.deployit.jcr.JcrConstants.ARTIFACT_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrUtils.countMatches;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.*;
import static com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot.NESTED;
import static com.xebialabs.deployit.repository.JcrPathHelper.getAbsolutePathFromId;
import static com.xebialabs.deployit.repository.JcrPathHelper.getIdFromAbsolutePath;
import static com.xebialabs.deployit.repository.NodeUtils.*;
import static java.util.EnumSet.of;
import static java.util.stream.Stream.concat;
import static scala.collection.JavaConversions.asScalaBuffer;

class JcrChangeSetExecutor implements JcrCallback<Object> {

    private final JcrRepositoryService jcrRepositoryService;
    private final ChangeSet changeset;
    private final PasswordEncrypter passwordEncrypter;
    private final LicenseService licenseService;
    private final ProgressLogger progressLogger;

    JcrChangeSetExecutor(JcrRepositoryService jcrRepositoryService, ChangeSet changeset, PasswordEncrypter passwordEncrypter, LicenseService licenseService, ProgressLogger progressLogger) {
        this.jcrRepositoryService = checkNotNull(jcrRepositoryService, "jcrRepositoryService");
        this.changeset = checkNotNull(changeset, "changeset");
        this.passwordEncrypter = checkNotNull(passwordEncrypter, "passwordencrypter");
        this.licenseService = licenseService;
        this.progressLogger = progressLogger;
    }

    @Override
    public Object doInJcr(Session session) throws RepositoryException {
        execute(session, false);
        return null;
    }

    void execute(Session session, boolean dryRun) throws RepositoryException {
        final LicenseTransaction licenseTransaction = licenseService.startTransaction();
        try {
            if (dryRun) {
                logger.debug("Not saving JCR checkpoints in dry run");
            } else if (changeset.isDisableVersionHistory()) {
                logger.debug("Not saving JCR checkpoints because version history has been disabled");
            } else {
                checkpointNodes(changeset, session);
            }
            createEntities(changeset, session, licenseTransaction);
            createOrUpdateEntities(changeset, session, licenseTransaction);
            updateEntities(changeset, session);
            moveEntities(changeset.getMoveCis(), session);
            renameEntities(changeset.getRenameCis(), session);
            copyEntities(changeset.getCopyCis(), session, licenseTransaction);

            // Fetches the history of the deleted items before deleting them. In case of not dry run the history will be removed together with nodes.
            SortedMap<String, VersionHistory> versionHistories = deleteEntities(changeset.getDeleteCiIds(), session, licenseTransaction);
            if (dryRun) {
                logger.debug("Not committing JCR session and not removing JCR version histories in dry run");
                licenseService.rollbackTransaction(licenseTransaction);
            } else {
                // when saving the session, the version histories are still retained, so we remove them 'manually' afterwards
                // also, deleting nodes without a history will result in exceptions.
                saveSession(session);
                removeVersionHistories(versionHistories);
            }
        } catch (ReferentialIntegrityException exc) {
            licenseService.rollbackTransaction(licenseTransaction);
            throw new ItemInUseException(exc, "Cannot delete configuration items [%s] because one of the configuration items, or one of their children, is still being referenced", Strings.mkString(changeset.getDeleteCiIds(), ","));
        } catch (Exception e) {
            licenseService.rollbackTransaction(licenseTransaction);
            throw e;
        }
    }

    private void checkpointNodes(ChangeSet changeset, final Session session) throws RepositoryException {
        logger.debug("Saving JCR checkpoints");
        if (hasUpdatesOrDeletes(changeset)) {
            this.progressLogger.log("Updating version history of CIs");
        }
        Predicate<ConfigurationItem> ciExists = ci -> {
            try {
                return session.nodeExists(getAbsolutePathFromId(ci.getId()));
            } catch (RepositoryException e) {
                throw new RuntimeRepositoryException(e.getMessage(), e);
            }
        };

        Predicate<ConfigurationItem> ciIsVersioned = ci -> ci.getType().getDescriptor().isVersioned();

        Stream<ConfigurationItem> updateCisFromCreateOrUpdate = changeset.getCreateOrUpdateCis().stream().filter(ciIsVersioned).filter(ciExists);
        Stream<ConfigurationItem> updateCis = changeset.getUpdateCis().stream().filter(ciIsVersioned);
        concat(updateCisFromCreateOrUpdate, updateCis).forEach((ci) -> {
            try {
                checkpoint(session.getNode(getAbsolutePathFromId(ci.getId())), session);
            } catch (RepositoryException e) {
                throw new RuntimeRepositoryException("Cannot checkpoint node", e);
            }
        });

        for (String id : changeset.getDeleteCiIds()) {
            // Create a checkpoint so that there is at least one version. Jackrabbit will only purge empty version histories that had at least one
            // non-baseVersion.
            if (session.nodeExists(getAbsolutePathFromId(id))) {
                Node node = session.getNode(getAbsolutePathFromId(id));
                deepCheckpoint(node, session);
            }
        }
    }

    private boolean hasUpdatesOrDeletes(ChangeSet changeset) {
        return !changeset.getUpdateCis().isEmpty() || !changeset.getDeleteCiIds().isEmpty() || !changeset.getCreateOrUpdateCis().isEmpty();
    }

    private void deepCheckpoint(Node node, Session session) throws RepositoryException {
        checkpoint(node, session);
        NodeIterator nodes = node.getNodes();
        while (nodes.hasNext()) {
            deepCheckpoint(nodes.nextNode(), session);
        }
    }

    private void checkpoint(Node node, Session session) throws RepositoryException {
        if (node.isNodeType(NodeType.MIX_VERSIONABLE)) {
            logger.debug("Checkpointing node [{}]", getIdFromAbsolutePath(node.getPath()));
            VersionManager versionManager = session.getWorkspace().getVersionManager();
            versionManager.checkpoint(node.getPath());
        }
    }

    private void createEntities(ChangeSet changeset, Session session, final LicenseTransaction licenseTransaction) throws RepositoryException {
        List<ConfigurationItem> createCis = changeset.getCreateCis();
        if (!createCis.isEmpty()) {
            logger.debug("Creating nodes...");
        }

        createNodes(createCis, session, licenseTransaction);
    }

    private void updateEntities(ChangeSet changeset, Session session) throws RepositoryException {
        List<ConfigurationItem> updateCis = changeset.getUpdateCis();
        if (!updateCis.isEmpty()) {
            logger.debug("Updating nodes...");
        }

        updateNodes(updateCis, session);
    }

    private void createOrUpdateEntities(ChangeSet changeset, Session session, LicenseTransaction licenseTransaction) throws RepositoryException {
        List<ConfigurationItem> createOrUpdateCis = changeset.getCreateOrUpdateCis();
        if (!createOrUpdateCis.isEmpty()) {
            logger.debug("Creating or updating nodes...");
        }
        List<ConfigurationItem> toBeCreated = new ArrayList<>();
        List<ConfigurationItem> toBeUpdated = new ArrayList<>();

        for (ConfigurationItem createOrUpdateCi : createOrUpdateCis) {
            if (!session.nodeExists(getAbsolutePathFromId(createOrUpdateCi.getId()))) {
                toBeCreated.add(createOrUpdateCi);
            } else {
                toBeUpdated.add(createOrUpdateCi);
            }
        }

        changeset.getCreateOrUpdateActualCreatedCis().addAll(toBeCreated);
        changeset.getCreateOrUpdateActualUpdatedCis().addAll(toBeUpdated);

        createNodes(toBeCreated, session, licenseTransaction);
        updateNodes(toBeUpdated, session);

    }

    private List<Node> retrieveNodes(Iterable<ConfigurationItem> cis, Session session) throws RepositoryException {
        List<Node> nodes = new ArrayList<>();
        for (ConfigurationItem ci : cis) {
            checkNotNull(ci, "ci is null");
            try {
                final Node node = session.getNode(getAbsolutePathFromId(ci.getId()));
                nodes.add(node);
            } catch (PathNotFoundException exc) {
                throw new NotFoundException("Repository entity [%s] not found", ci.getId());
            }
        }
        return nodes;
    }

    private void createNodes(List<ConfigurationItem> createCis, Session session, final LicenseTransaction licenseTransaction) throws RepositoryException {
        verifyEntitiesMeetCreationPreconditions(createCis);
        Collections.sort(createCis, new SortHierarchyComparator());

        licenseService.getCounter().registerCisCreation(asScalaBuffer(createCis), licenseTransaction);

        List<NodeWriter> writers = new ArrayList<>();
        for (ConfigurationItem entity : createCis) {
            this.progressLogger.log("Creating %s of type %s", entity.getId(), entity.getType());
            Node node = createNode(entity, session);
            final NodeWriter nodeWriter = new NodeWriter(session, entity, node, passwordEncrypter);
            nodeWriter.writeBasics();
            writers.add(nodeWriter);
        }

        for (NodeWriter writer : writers) {
            writer.write();
        }
    }

    private Node createNode(ConfigurationItem entity, Session session) throws RepositoryException {
        validateNodeStoredInCorrectPath(entity.getId(), entity.getType(), session);
        logger.debug("Creating node [{}]", entity.getId());

        final Node node;
        node = session.getRootNode().addNode(entity.getId());
        int index = node.getIndex();
        if (index != 1) {
            node.remove();
            throw new ItemAlreadyExistsException("A Configuration Item with ID [%s] already exists.", entity.getId());
        }
        setMixins(entity, node);
        return node;
    }

    private void updateNodes(List<ConfigurationItem> updateCis, Session session) throws RepositoryException {
        List<Node> nodes = retrieveNodes(updateCis, session);
        for (int i = 0; i < updateCis.size(); i++) {
            updateNode(updateCis.get(i), nodes.get(i), session);
        }
    }

    private void updateNode(ConfigurationItem entity, Node node, Session session) throws RepositoryException {
        logger.debug("Updating node [{}]", entity.getId());
        this.progressLogger.log("Updating %s of type %s", entity.getId(), entity.getType());
        new NodeWriter(session, entity, node, passwordEncrypter).write();
    }


    private SortedMap<String, VersionHistory> deleteEntities(List<String> entityIds, Session session, LicenseTransaction licenseTransaction) throws RepositoryException {

        entityIds.sort(new SortEntityIdsHierarchyComparator().reversed());
        List<Node> nodes = retrieveNodesToBeDeleted(entityIds, session);
        SortedMap<String, VersionHistory> versionHistories = new TreeMap<>();
        for (Node node : nodes) {
            retrieveVersionHistories(node, session, versionHistories);
        }

        for (Node node : nodes) {
            logger.debug("Deleting node [{}]", getIdFromAbsolutePath(node.getPath()));
            this.progressLogger.log("Deleting %s", getIdFromAbsolutePath(node.getPath()));
            removeParentListCiReferences(node, session);
            Type type = typeOf(node);
            node.remove();
            licenseService.getCounter().registerTypeRemoval(type, licenseTransaction);
        }

        return versionHistories;
    }

    private void removeParentListCiReferences(Node node, Session session) throws RepositoryException {
        Node parent = node.getParent();
        Type t = typeOf(parent);
        PropertyDescriptor containmentListProperty = findContainmentListProperty(t, typeOf(node));
        if (containmentListProperty == null) {
            return;
        }

        Property property = parent.getProperty(containmentListProperty.getName());
        Value[] values = property.getValues();
        List<Value> newValues = new ArrayList<>();
        for (Value value : values) {
            Node referencedCiNode = getReferencedCiNode(parent, value, session);
            if (node.getPath().equals(referencedCiNode.getPath())) {
                continue;
            }
            newValues.add(session.getValueFactory().createValue(referencedCiNode));
        }
        parent.setProperty(containmentListProperty.getName(), newValues.toArray(new Value[newValues.size()]));
    }

    private List<Node> retrieveNodesToBeDeleted(List<String> ids, Session session) throws RepositoryException {
        List<Node> nodes = new ArrayList<>();
        for (String id : ids) {
            checkNotNull(id, "id is null");
            logger.debug("Retrieving node {} to delete it.", id);
            try {
                Node node = session.getNode(getAbsolutePathFromId(id));
                nodes.add(node);
            } catch (PathNotFoundException ignored) {
                //ignore
            }
        }
        return nodes;
    }

    private void retrieveVersionHistories(Node node, Session session, SortedMap<String, VersionHistory> versionHistories) throws RepositoryException {
        if (node.isNodeType(NodeType.MIX_VERSIONABLE)) {
            final String id = getIdFromAbsolutePath(node.getPath());
            logger.debug("Retrieving version history for node {} to delete it.", id);
            final VersionHistory versionHistory = session.getWorkspace().getVersionManager().getVersionHistory(node.getPath());
            versionHistories.put(id, versionHistory);
        }
        NodeIterator nodes = node.getNodes();
        while (nodes.hasNext()) {
            Node each = nodes.nextNode();
            retrieveVersionHistories(each, session, versionHistories);
        }
    }

    private void renameEntities(List<Tuple<String, String>> renameCis, Session session) throws RepositoryException {
        if (!renameCis.isEmpty()) {
            logger.debug("Renaming nodes...");
        }

        for (Tuple<String, String> renameCi : renameCis) {
            String oldId = renameCi.getA();
            String newId = oldId.substring(0, oldId.lastIndexOf('/') + 1) + renameCi.getB();
            Type t = JcrRepositoryService.readType(session, oldId);
            if (t.equals(Type.valueOf(Root.class))) {
                throw new DeployitException("Cannot rename a core.Root configuration item");
            }

            moveNode(getAbsolutePathFromId(oldId), getAbsolutePathFromId(newId), session);
        }
    }

    private void moveEntities(List<Tuple<String, String>> moveCis, Session session) throws RepositoryException {
        if (!moveCis.isEmpty()) {
            logger.debug("Moving nodes...");
        }

        for (Tuple<String, String> moveCi : moveCis) {
            String toBeMoved = moveCi.getA();
            String newId = moveCi.getB();
            checkArgument(!toBeMoved.equals(newId), "Cannot move ci [%s] to same location", newId);
            Type t = JcrRepositoryService.readType(session, toBeMoved);
            jcrRepositoryService.checkMoveAllowed(t);

            validateNodeStoredInCorrectPath(newId, t, session);
            moveNode(getAbsolutePathFromId(toBeMoved), getAbsolutePathFromId(newId), session);
        }
    }

    private void moveNode(String fromId, String toId, Session session) throws RepositoryException {
        logger.debug("Moving/renaming node [{}] to [{}]", fromId, toId);
        if (session.nodeExists(toId)) {
            throw new ItemAlreadyExistsException("The destination id [%s] exists.", toId);
        }
        session.move(fromId, toId);
    }

    private void copyEntities(List<Tuple<String, String>> copyCis, final Session session, final LicenseTransaction licenseTransaction) throws RepositoryException {
        if (!copyCis.isEmpty()) {
            logger.debug("Copying nodes...");
        }

        Seq<Type> types = new RepositoryUtils(jcrRepositoryService).readTree(copyCis);

        licenseService.getCounter().registerTypesCreation(types, licenseTransaction);

        for (Tuple<String, String> copyCi : copyCis) {
            String toBeCopied = copyCi.getA();
            String newId = copyCi.getB();
            checkArgument(!toBeCopied.equals(newId), "Cannot copy ci [%s] to same location", newId);
            checkArgument(!Type.valueOf(Root.class).equals(JcrRepositoryService.readType(session, toBeCopied)), "Cannot copy root node [%s].", toBeCopied);

            jcrRepositoryService.checkCopyAllowed(session, toBeCopied, newId);
            copyNode(getAbsolutePathFromId(toBeCopied), getAbsolutePathFromId(newId), session);
        }
    }

    private void copyNode(String fromId, String toId, Session session) throws RepositoryException {
        logger.debug("Copy node [{}] to [{}]", fromId, toId);
        if (session.nodeExists(toId)) {
            throw new ItemAlreadyExistsException("The destination id [%s] exists.", toId);
        }
        session.getWorkspace().copy(fromId, toId);
    }

    private void saveSession(Session session) throws RepositoryException {
        logger.debug("Committing JCR session");
        this.progressLogger.log("Saving data to repository");
        session.save();
        this.progressLogger.log("Done");
    }

    private void removeVersionHistories(SortedMap<String, VersionHistory> versionHistories) throws RepositoryException {
        logger.debug("Removing version histories that have become obsolete because their corresponding nodes or one of their ancestors was deleted.");
        for (Map.Entry<String, VersionHistory> entry : versionHistories.entrySet()) {
            removeVersionHistory(entry.getKey(), entry.getValue());
        }
    }

    private void removeVersionHistory(String id, VersionHistory versionHistory) throws RepositoryException {
        logger.debug("Removing version history for node {}.", id);
        final VersionIterator allVersions = versionHistory.getAllVersions();
        Version rootVersion = versionHistory.getRootVersion();
        while (allVersions.hasNext()) {
            Version version = allVersions.nextVersion();
            if (!version.getName().equals(rootVersion.getName())) {
                logger.debug("Removing version {} from version history for node {}.", version.getName(), id);
                versionHistory.removeVersion(version.getName());
            }
        }
    }

    private void verifyEntitiesMeetCreationPreconditions(List<ConfigurationItem> cis) {
        for (int i = 0; i < cis.size(); i++) {
            ConfigurationItem ci = cis.get(i);
            checkNotNull(ci, "ci at index %s is null.", i);
            checkNotNull(ci.getId(), "ci at index %s has null id.", i);

            if (ci instanceof SourceArtifact) {
                SourceArtifact sourceArtifact = (SourceArtifact) ci;
                boolean hasCorrectFileUri = sourceArtifact.getFileUri() != null && ArtifactResolverRegistry.validate(sourceArtifact);
                checkArgument(hasRealOrResolvedFile(sourceArtifact) || hasCorrectFileUri, "Artifact %s should have either file or correct fileUri set", ci.getId());
            }
        }
    }

    private void validateNodeStoredInCorrectPath(String id, Type type, Session session) throws RepositoryException {
        checkArgument(DescriptorRegistry.exists(type), "Unknown configuration item type %s", type);

        final Descriptor desc = DescriptorRegistry.getDescriptor(type);

        final String[] pathElements = id.split("/");

        if (desc.isAssignableTo(Directory.class)) {
            checkArgument(pathElements.length >= 2, "A core.Directory must be stored under a Root node, not under %s.", id);
            for (String rootName : CiRoots.all()) {
                if (rootName.equals(pathElements[0])) {
                    return;
                }
            }
            throw new Checks.IncorrectArgumentException("A core.Directory must be stored under a valid Root node, not under %s.", id);
        } else if (desc.getRoot() != NESTED) {
            checkArgument(pathElements[0].equals(desc.getRootName()), "Configuration item of type %s cannot be stored at %s. It should be stored under %s", type, id, desc.getRootName());
        } else {
            checkArgument(pathElements.length >= 3, "Configuration item of type %s cannot be stored at %s. It should be stored under a valid parent node", type, id);
            final String parentId = id.substring(0, id.lastIndexOf('/'));
            final Node parentNode = session.getNode(getAbsolutePathFromId(parentId));
            checkArgument(parentNode != null, "Configuration item of type %s cannot be stored at %s. The parent does not exist", type, id);
            checkArgument(parentNode.isNodeType(CONFIGURATION_ITEM_NODETYPE_NAME), "Configuration item of type %s cannot be stored at %s. The parent is not a configuration item", type, id);
            final String parentTypeName = parentNode.getProperty(JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME).getString();
            final Descriptor parentDescriptor = DescriptorRegistry.getDescriptor(parentTypeName);

            // Check parent relation
            for (PropertyDescriptor aPD : desc.getPropertyDescriptors()) {
                if (aPD.getKind() == CI && aPD.isAsContainment()) {
                    if (parentDescriptor.isAssignableTo(aPD.getReferencedType())) {
                        return;
                    }
                }
            }

            // Check child relation
            for (PropertyDescriptor aPD : parentDescriptor.getPropertyDescriptors()) {
                if (of(SET_OF_CI, LIST_OF_CI).contains(aPD.getKind()) && aPD.isAsContainment()) {
                    if (desc.isAssignableTo(aPD.getReferencedType())) {
                        return;
                    }
                }
            }

            throw new Checks.IncorrectArgumentException("Configuration item of type %s cannot be stored at %s. The parent cannot contain configuration items of this type", type, id);
        }
    }

    private void setMixins(ConfigurationItem entity, Node node)
            throws RepositoryException {
        if (entity instanceof Artifact) {
            node.addMixin(ARTIFACT_NODETYPE_NAME);
            node.addMixin(CONFIGURATION_ITEM_NODETYPE_NAME);
        } else {
            node.addMixin(CONFIGURATION_ITEM_NODETYPE_NAME);
        }
        node.addMixin(NodeType.MIX_REFERENCEABLE);
        if (entity.getType().getDescriptor().isVersioned()) {
            node.addMixin(NodeType.MIX_VERSIONABLE);
        }
    }

    private static class SortHierarchyComparator implements Comparator<ConfigurationItem> {
        @Override
        public int compare(ConfigurationItem o1, ConfigurationItem o2) {
            final int nrSlashes1 = countMatches(o1.getId(), '/');
            final int nrSlashes2 = countMatches(o2.getId(), '/');
            return nrSlashes1 - nrSlashes2;
        }
    }

    private static class SortEntityIdsHierarchyComparator implements Comparator<String> {
        @Override
        public int compare(String s1, String s2) {
            final int nrSlashes1 = countMatches(s1, '/');
            final int nrSlashes2 = countMatches(s2, '/');
            return nrSlashes1 - nrSlashes2;
        }
    }

    private static Logger logger = LoggerFactory.getLogger(JcrChangeSetExecutor.class);

}
