package com.xebialabs.deployit.repository;

import com.xebialabs.deployit.jcr.JcrCallback;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.license.service.LicenseService;

import javax.jcr.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.repository.JcrPathHelper.getAbsolutePathFromId;
import static com.xebialabs.deployit.repository.NodeUtils.descriptorOf;

class JcrReferentialIntegrityChecker implements JcrCallback<Object> {

    private final JcrRepositoryService jcrRepositoryService;
    private final ChangeSet changeset;
    private final PasswordEncrypter passwordEncrypter;
    private final LicenseService licenseService;

    JcrReferentialIntegrityChecker(JcrRepositoryService jcrRepositoryService, ChangeSet changeset, PasswordEncrypter passwordEncrypter, LicenseService licenseService) {
        this.jcrRepositoryService = checkNotNull(jcrRepositoryService, "jcrRepositoryService");
        this.changeset = checkNotNull(changeset, "changeset");
        this.passwordEncrypter = checkNotNull(passwordEncrypter, "passwordEncrypter");
        this.licenseService = licenseService;
    }

    @Override
    public Object doInJcr(Session session) throws RepositoryException {
        checkIfAnyOfTheEntitiesToBeCreatedAlreadyExists(session);
        Map<String, List<PropertyRef>> deletedEntityRefs = findReferencesToDeletedNodes(session);
        JcrChangeSetExecutor changeSetExecutor = new JcrChangeSetExecutor(jcrRepositoryService, changeset, passwordEncrypter, licenseService, new NullProgressLogger());
        changeSetExecutor.execute(session, true);
        try {
            checkForRemainingReferencesToDeletedNodes(deletedEntityRefs, session);
        } finally {
            session.refresh(false);
        }
        return null;
    }

    private void checkIfAnyOfTheEntitiesToBeCreatedAlreadyExists(Session session) throws RepositoryException {
        for (ConfigurationItem entity : changeset.getCreateCis()) {
            if (session.nodeExists(getAbsolutePathFromId(entity.getId()))) {
                throw new ItemAlreadyExistsException("Repository entity %s already exists", entity.getId());
            }
        }
    }

    private Map<String, List<PropertyRef>> findReferencesToDeletedNodes(Session session) throws RepositoryException {

        Map<String, List<PropertyRef>> deletedEntityRefs = new HashMap<>();
        for (String deleteNodeId : changeset.getDeleteCiIds()) {
            if (!session.nodeExists(getAbsolutePathFromId(deleteNodeId))) {
                continue;
            }
            Node node = session.getNode(getAbsolutePathFromId(deleteNodeId));
            PropertyIterator references = node.getReferences();
            while (references.hasNext()) {
                Property property = references.nextProperty();
                Node parent = property.getParent();
                if (isParentListCiReference(property, node)) {
                    continue;
                }
                deletedEntityRefs.computeIfAbsent(deleteNodeId, (i) -> new ArrayList<>()).add(new PropertyRef(parent.getPath(), property.getName(), node.getIdentifier()));
            }
        }
        return deletedEntityRefs;
    }

    private boolean isParentListCiReference(Property referenceProperty, Node nodeToBeDeleted) throws RepositoryException {
        Node owningNode = referenceProperty.getParent();
        if (!nodeToBeDeleted.getParent().getPath().equals(owningNode.getPath())) {
            return false;
        }

        final PropertyDescriptor propertyDescriptor = descriptorOf(referenceProperty);
        return propertyDescriptor.isAsContainment() && propertyDescriptor.getKind() == LIST_OF_CI;
    }

    private void checkForRemainingReferencesToDeletedNodes(Map<String, List<PropertyRef>> deletedEntityRefs, Session session) throws RepositoryException {
        for (String deleteNodeId : deletedEntityRefs.keySet()) {
            for (PropertyRef propertyRef : deletedEntityRefs.get(deleteNodeId)) {
                if (session.nodeExists(propertyRef.owningNode)) {
                    Node owningNode = session.getNode(propertyRef.owningNode);
                    if (owningNode.hasProperty(propertyRef.property)) {
                        Property property = owningNode.getProperty(propertyRef.property);
                        Value[] values = property.isMultiple() ? property.getValues() : new Value[]{property.getValue()};
                        for (Value value : values) {
                            if (value.getString().equals(propertyRef.uuidOfReferencedNode)) {
                                throw new ItemInUseException("Repository entity %s is still referenced by %s", deleteNodeId, getOwnerNodeWithoutLeadingSlash(propertyRef));
                            }
                        }
                    }
                }
            }
        }
    }

    private String getOwnerNodeWithoutLeadingSlash(PropertyRef propertyRef) {
        String owningNode = propertyRef.owningNode;
        if (owningNode.charAt(0) == '/') {
            return owningNode.substring(1);
        }
        return owningNode;
    }


    private static class PropertyRef {
        private String owningNode;
        private String property;
        private String uuidOfReferencedNode;

        PropertyRef(String owningNode, String property, String uuidOfReferencedNode) {
            this.owningNode = owningNode;
            this.property = property;
            this.uuidOfReferencedNode = uuidOfReferencedNode;
        }
    }

}
