package com.xebialabs.deployit.repository;

import java.util.*;
import java.util.stream.Collectors;
import javax.jcr.*;
import javax.jcr.nodetype.NodeType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.core.ListOfStringView;
import com.xebialabs.deployit.core.MapStringStringView;
import com.xebialabs.deployit.core.SetOfStringView;
import com.xebialabs.deployit.core.StringValue;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.engine.spi.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.io.SourceArtifactFile;
import com.xebialabs.deployit.jcr.JcrConstants;
import com.xebialabs.deployit.jcr.JcrUtils;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.util.PasswordEncrypter;

import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.FILENAME_PROPERTY_NAME;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.MAP_STRING_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_STRING;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.STRING;
import static com.xebialabs.deployit.repository.JcrPathHelper.getIdFromAbsolutePath;
import static com.xebialabs.deployit.repository.StringValueConverter.stringToValue;
import static java.lang.String.format;

class NodeReader {
    private Session session;
    private Node node;
    private final WorkDir workDir;
    private NodeReaderContext context;
    private final PasswordEncrypter passwordEncrypter;

    static <T extends ConfigurationItem> T read(Session session, Node node, int depth, WorkDir workDir, PasswordEncrypter passwordEncrypter) throws RepositoryException {
        NodeReaderContext c = NodeReaderContext.get().hold();
        try {
            return NodeReader.read(session, node, depth, workDir, c, passwordEncrypter);
        } finally {
            c.release();
        }
    }

    static <T extends ConfigurationItem> T read(Session session, Node node, int depth, WorkDir workDir, NodeReaderContext context, PasswordEncrypter passwordEncrypter) throws RepositoryException {
        return new NodeReader(session, node, workDir, context, passwordEncrypter).read(depth);
    }

    private NodeReader(Session session, Node node, WorkDir workDir, NodeReaderContext context, PasswordEncrypter passwordEncrypter) throws RepositoryException {
        logger.trace("Creating NodeReader for CI [{}] with workDir {} and context {}", getIdFromAbsolutePath(node.getPath()), workDir, context);
        this.session = session;
        this.node = node;
        this.workDir = workDir;
        this.context = context;
        this.passwordEncrypter = passwordEncrypter;
    }

    @SuppressWarnings({"unchecked"})
    private <T extends ConfigurationItem> T read(int depth) throws RepositoryException {
        String id = getIdFromAbsolutePath(node.getPath());
        logger.trace("Reading node [{}] with depth {} and workdir {}", id, depth, workDir);
        T item = this.checkAlreadyRead();
        if (item != null) {
            return item;
        }

        final Type type = readType(node);
        if (!type.exists()) {
            logger.warn("While trying to read node [{}] its type [{}] was not found in any plugin. Please make sure the required plugin is installed.", node.getPath(), type);
            throw new TypeNotFoundException(format("Unknown type [%s] while reading node [%s]", type, node.getPath()));
        }
        Descriptor descriptor = DescriptorRegistry.getDescriptor(type);

        item = descriptor.newInstance(id);

        if(depth > 0) {
            if (item instanceof BaseConfigurationItem) {
                ((BaseConfigurationItem) item).set$ciAttributes(NodeUtils.readCiAttributes(node));

                if (node.hasProperty(JcrConstants.TOKEN_PROPERTY_NAME)) {
                    ((BaseConfigurationItem) item).set$token(node.getProperty(JcrConstants.TOKEN_PROPERTY_NAME).getString());
                }
            }

            // Only store the item in the cache if we read it fully.
            context.put(node.getIdentifier(), item);
            copyValues(item, depth - 1, descriptor);
            if(item instanceof SourceArtifact) {
                setLazySourceFile((SourceArtifact) item);
            }
        }
        return item;
    }

    static Type readType(final Node node) throws RepositoryException {
        if (node.hasProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME)) {
            Property property = node.getProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME);
            return Type.valueOf(property.getString());
        } else {
            return null;
        }
    }

    @SuppressWarnings({ "rawtypes", "unchecked" })
    private <T extends ConfigurationItem> T checkAlreadyRead() throws RepositoryException {
        if (context.hasItem(node.getIdentifier())) {
            T item = (T) context.get(node.getIdentifier());
            if (item instanceof SourceArtifact) {
                setLazySourceFile((SourceArtifact) item);
            }
            return item;
        }
        return null;
    }

    private void setLazySourceFile(SourceArtifact sourceArtifact) throws RepositoryException {
        String filename = node.getProperty(FILENAME_PROPERTY_NAME).getString();
        sourceArtifact.setFile(SourceArtifactFile.withNullableWorkDir(filename, sourceArtifact, workDir));
    }

    private <T extends ConfigurationItem> void copyValues(T item, int depth, Descriptor descriptor) throws RepositoryException {
        logger.trace("Reading properties for CI [{}]", item.getId());
        for (PropertyDescriptor pd : descriptor.getPropertyDescriptors()) {
            if (!hasProperty(node, pd) && !pd.isAsContainment()) {
                logger.trace("Repository node [{}] does not contain value for (non-containment) property [{}]. Using the default value.", item.getId(), pd);
                wrapDefaultValue(item, pd);
                continue;
            }

            if (pd.isTransient()) {
                if (hasProperty(node, pd)) {
                    logger.debug("Repository node [{}] contains transient property [{}] which should not have been persisted.", item.getId(), pd);
                }
                logger.trace("Not attempting to read transient property [{}] from repository node [{}]. Using the default value.", pd, item.getId());
                wrapDefaultValue(item, pd);
                continue;
            }
            logger.trace("Reading property [{}]", pd.getFqn());

            switch (pd.getKind()) {
                case BOOLEAN:
                case INTEGER:
                case STRING:
                case ENUM:
                    setPrimitiveProperty(item, pd);
                    break;
                case DATE:
                    pd.set(item, node.getProperty(pd.getName()).getDate().getTime());
                    break;
                case SET_OF_STRING:
                    SetOfStringView setOfStringView = new SetOfStringView(new HashSet<>(getCollectionOfStringValues(pd)));
                    if (pd.isPassword()) {
                        setOfStringView = setOfStringView.encrypt();
                    }
                    pd.set(item, setOfStringView);
                    break;
                case SET_OF_CI:
                    pd.set(item, new HashSet<>(getCollectionOfConfigurationItemValues(item, depth, pd)));
                    break;
                case LIST_OF_STRING:
                    ListOfStringView listOfStringView = new ListOfStringView(new ArrayList<>(getCollectionOfStringValues(pd)));
                    if (pd.isPassword()) {
                        listOfStringView = listOfStringView.encrypt();
                    }
                    pd.set(item, listOfStringView);
                    break;
                case LIST_OF_CI:
                    pd.set(item, new ArrayList<>(getCollectionOfConfigurationItemValues(item, depth, pd)));
                    break;
                case CI:
                    setConfigurationItemProperty(item, depth, pd);
                    break;
                case MAP_STRING_STRING:
                    copyMapPropertyFromNode(item, pd);
                    break;
                default:
                    throw new IllegalStateException("Unsupported property kind: " + pd.getKind());
            }
        }
    }

    @SuppressWarnings("unchecked")
    private <T extends ConfigurationItem> void wrapDefaultValue(T item, PropertyDescriptor pd) {
        Object value = pd.get(item);
        Object newValue;
        // Wrap the default values in String Views.
        if (pd.getKind() == SET_OF_STRING) {
            newValue = SetOfStringView.from((Set<String>) value);
        } else if (pd.getKind() == LIST_OF_STRING) {
            newValue = ListOfStringView.from((List<String>) value);
        } else if (pd.getKind() == MAP_STRING_STRING) {
            newValue = MapStringStringView.from((Map<String, String>) value);
        } else {
            return;
        }
        logger.trace("Wrapping default value [{}] in property [{}] of CI [{}]", value, pd.getFqn(), item.getId());
        pd.set(item, newValue);
    }

    private static boolean hasProperty(Node node, PropertyDescriptor pd) throws RepositoryException {
        if (node.hasProperty(pd.getName())) return true;
        for (String s : pd.getAliases()) {
            if (node.hasProperty(s)) return true;
        }
        return false;
    }

    private static Property getProperty(Node node, PropertyDescriptor pd) throws RepositoryException {
        if (node.hasProperty(pd.getName())) return node.getProperty(pd.getName());
        for (String s : pd.getAliases()) {
            if (node.hasProperty(s)) return node.getProperty(s);
        }
        throw new IllegalStateException("Should have found a property on the node " + node.getPath() + " for propertydescriptor " + pd.getFqn());
    }

    private <T extends ConfigurationItem> void copyMapPropertyFromNode(T item, PropertyDescriptor pd) throws RepositoryException {
        Property property = getProperty(node, pd);
        Map<String, String> value = JcrUtils.readMap(property);
        MapStringStringView mapStringStringView = new MapStringStringView(value.entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, (entry) -> stringToValue(passwordEncrypter).apply(entry.getValue()))));
        if (pd.isPassword()) {
            mapStringStringView = mapStringStringView.encrypt();
        }
        pd.set(item, mapStringStringView);
    }

    private <T extends ConfigurationItem> void setConfigurationItemProperty(T item, int depth, PropertyDescriptor pd) throws RepositoryException {
        Node referencedNode;
        if (pd.isAsContainment()) {
            if (node.isNodeType(NodeType.NT_FROZEN_NODE)) {
//                Getting the parent node doesn't work in case this is a frozen node, the parent of a frozennode is the versionhistory.
                logger.debug("Reading parent of frozen node [{}]", node.getPath());
                Node realNode = session.getNodeByIdentifier(node.getProperty(Property.JCR_FROZEN_UUID).getString());
                referencedNode = realNode.getParent();
            } else {
                referencedNode = node.getParent();
            }
        } else {
            Value value = getProperty(node, pd).getValue();
            referencedNode = NodeUtils.getReferencedCiNode(node, value, session);
        }
        pd.set(item, readReference(referencedNode, depth));
    }

    private <T extends ConfigurationItem> Collection<ConfigurationItem> getCollectionOfConfigurationItemValues(T item, int depth, PropertyDescriptor pd) throws RepositoryException {
        Collection<ConfigurationItem> items = new ArrayList<>();
        if (pd.isAsContainment() && pd.getKind() == SET_OF_CI) {
            NodeIterator nodes = node.getNodes();
            while (nodes.hasNext()) {
                Node ref = nodes.nextNode();
                Type refType = readType(ref);
                if (refType != null && refType.instanceOf(pd.getReferencedType())) {
                    items.add(readReference(ref, depth));
                }
            }
        } else if (pd.isAsContainment() && pd.getKind() == LIST_OF_CI && !hasProperty(node, pd)) {
            return items;
        } else {
            for (Value each : getProperty(node, pd).getValues()) {
                Node referencedNode = NodeUtils.getReferencedCiNode(node, each, session);
                items.add(readReference(referencedNode, depth));
            }
        }
        return items;
    }

    private ConfigurationItem readReference(Node referencedNode, int depth) throws RepositoryException {
        return NodeReader.read(session, referencedNode, depth, workDir, context, passwordEncrypter);
    }

    private Collection<StringValue> getCollectionOfStringValues(PropertyDescriptor pd) throws RepositoryException {
        final StringValueConverter converter = new StringValueConverter(passwordEncrypter);
        Collection<StringValue> list = new ArrayList<>();
        Property property = getProperty(node, pd);
        if(property.isMultiple()){
            for (Value v : property.getValues()) {
                list.add(converter.convert(v.getString()));
            }
        } else {
            list.add(converter.convert(property.getString()));
        }
        return list;
    }

    private <T extends ConfigurationItem> void setPrimitiveProperty(T item, PropertyDescriptor pd) throws RepositoryException {
        String valueAsString = getProperty(node, pd).getString();
        if (pd.getKind() == STRING && pd.isPassword()) {
            valueAsString = passwordEncrypter.ensureDecrypted(valueAsString);
        }
        pd.set(item, valueAsString);
    }

    private static final Logger logger = LoggerFactory.getLogger(NodeReader.class);

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 500)
    public static class TypeNotFoundException extends DeployitException {
        public TypeNotFoundException(final String message) {
            super(message);
        }
    }
}
