package com.xebialabs.xlplatform.artifact.resolution.jcr

import java.io.{IOException, InputStream}
import java.net.URI
import javax.jcr.Session

import com.xebialabs.deployit.engine.spi.artifact.resolution.ArtifactResolver.Resolver
import com.xebialabs.deployit.engine.spi.artifact.resolution.{ArtifactResolver, ResolvedArtifactFile}
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.jcr.JcrConstants._
import com.xebialabs.deployit.jcr.{RuntimeRepositoryException, JcrCallback, JcrTemplateHolder}
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact
import com.xebialabs.deployit.repository.JcrPathHelper

import scala.util.{Failure, Success, Try}

object JcrArtifactResolver {
  val Protocol = "jcr"

  def canHandle(uri: String) = URI.create(uri).getScheme == Protocol
}

@Resolver(protocols = Array("jcr"))
class JcrArtifactResolver extends ArtifactResolver {

  /**
   * JCR URI's are of the form 'jcr:file.txt' (i.e. 'jcr' protocol, and only the filename in the scheme specific part.
   **/
  override def resolveLocation(artifact: SourceArtifact): ResolvedArtifactFile = new ResolvedArtifactFile {
    override def getFileName: String = getFileUri(artifact).getSchemeSpecificPart

    override def close(): Unit = {}

    override def openStream: InputStream = {
      Try(JcrTemplateHolder.get().execute(new JcrCallback[InputStream] {
        override def doInJcr(session: Session): InputStream = {
          val id: String = artifact.getId
          val binary = session.getNode(JcrPathHelper.getAbsolutePathFromId(id)).getProperty(DATA_PROPERTY_NAME).getBinary
          new JcrDelegatingInputStream(binary)
        }
      }, true)) match {
        case Failure(exception: RuntimeRepositoryException) =>
          throw new NotFoundException(s"Artifact data not found for: ${artifact.getId}")
        case Failure(exception: IOException) => throw exception
        case Failure(ex) => throw new IOException(ex)
        case Success(value) => value
      }
    }

  }

  override def validateCorrectness(artifact: SourceArtifact): Boolean = getFileUri(artifact).getScheme == JcrArtifactResolver.Protocol

  private def getFileUri(artifact: SourceArtifact) = URI.create(artifact.getFileUri)
}
