package com.xebialabs.xlplatform.repository

import java.util.Calendar
import javax.jcr._
import javax.jcr.nodetype.{NodeType, NodeTypeManager, NodeTypeTemplate}

import com.xebialabs.deployit.JcrCredentials
import com.xebialabs.deployit.booter.local.CiRoots
import com.xebialabs.deployit.jcr.JcrConstants._
import com.xebialabs.deployit.jcr.JcrTemplate
import com.xebialabs.deployit.repository.internal.Root
import grizzled.slf4j.Logging

import scala.collection.convert.wrapAll._
import scala.util.Try

class XlJcrRepositoryInitializer(xlRepositoryConfig: XlRepositoryConfig) extends Logging {

  val jcrCredentials = xlRepositoryConfig.credentials match {
    case JcrCredentials(creds) => creds
    case _ => throw new IllegalStateException("Credentials must be of type JcrCredentials")
  }

  val repositoryId = xlRepositoryConfig.repositoryId

  def init(repository: Repository): Unit = {
    implicit val session: Session = repository.login(jcrCredentials)
    try {
      if (isInitialized(session)) {
        logger.info("Nodetypes already initialized!")
      } else {
        initializeNodeTypes(session)
      }
      doInitialize(repository)
    } finally {
      session.logout()
    }
  }

  private[this] def initializeNodeTypes(implicit session: Session): Unit = {
    session.getWorkspace.getNamespaceRegistry.registerNamespace(DEPLOYIT_NAMESPACE_PREFIX, DEPLOYIT_NAMESPACE_URI)

    Seq(CONFIGURATION_ITEM_NODETYPE_NAME,
      ARTIFACT_NODETYPE_NAME,
      TASK_NODETYPE_NAME,
      STEP_NODETYPE_NAME,
      CONFIGURATION_NODETYPE_NAME).foreach(createMixinType)
  }

  private[this] def doInitialize(repository: Repository)(implicit session: Session): Unit = {

    require(CiRoots.all().nonEmpty, "No ConfigurationItems Roots are found in type system")
    CiRoots.all().filter(_.nonEmpty).foreach(createXlRoot)

    if (!existsNode(CONFIGURATION_NODE_NAME)) {
      createNode(CONFIGURATION_NODE_NAME).addMixin(CONFIGURATION_NODETYPE_NAME)
    }

    Seq(TASKS_NODE_NAME, SECURITY_NODE_NAME, ROLES_NODE_NAME, ROLE_ASSIGNMENTS_NODE_NAME).foreach(createNodeIfNotExists)

    storeRepoVersion
    // metadata service used in storeRepositoryId might create a new session - we have to save before it's invoked
    session.save()

    storeRepositoryId(repository)
    session.save()
  }

  private[this] def isInitialized(session: Session): Boolean = {
    Try(session.getNamespaceURI(DEPLOYIT_NAMESPACE_PREFIX))
      .map(_ => true)
      .recover({case nse: NamespaceException => false})
      .get
  }

  private[this] def createMixinType(typeName: String)(implicit session: Session) = {
    val manager: NodeTypeManager = session.getWorkspace.getNodeTypeManager
    val nodeTypeTemplate: NodeTypeTemplate = manager.createNodeTypeTemplate
    nodeTypeTemplate.setName(typeName)
    nodeTypeTemplate.setQueryable(true)
    nodeTypeTemplate.setAbstract(false)
    nodeTypeTemplate.setMixin(true)
    val nodeType: NodeType = manager.registerNodeType(nodeTypeTemplate, false)
    logger.debug(s"Registered NodeType definition: $nodeType")
  }

  private[this] def createXlRoot(rootName: String)(implicit session: Session) = {
    if (!existsNode(rootName)) {
      val node: Node = createNode(rootName)
      node.addMixin(CONFIGURATION_ITEM_NODETYPE_NAME)
      node.addMixin(NodeType.MIX_VERSIONABLE)
      node.setProperty(CONFIGURATION_ITEM_TYPE_PROPERTY_NAME, "internal." + classOf[Root].getSimpleName)
      node.setProperty(LAST_MODIFIED_AT_PROPERTY_NAME, Calendar.getInstance)
      logger.debug(s"Configured XL Rootnode $rootName at $node")
    }
  }

  private[this] def existsNode(nodeName: String)(implicit session: Session) = {
    session.getRootNode.hasNode(nodeName)
  }

  private[this] def createNodeIfNotExists(nodeName: String)(implicit session: Session) = {
    if (!existsNode(nodeName)) {
      createNode(nodeName)
    }
  }

  private[this] def createNode(nodeName: String)(implicit session: Session) = {
    val node: Node = session.getRootNode.addNode(nodeName)
    logger.debug(s"Created node $node")
    node
  }

  private[this] def storeRepoVersion(implicit session: Session): Unit = {
    if (!existsNode(VERSIONS_NODE_NAME)) {
      val node: Node = session.getRootNode.addNode(VERSIONS_NODE_NAME)
      logger.debug(s"Initialized repository version node at $node")
    }
  }

  private[this] def storeRepositoryId(repository: Repository) = {
    val template = new JcrTemplate(repository, jcrCredentials)
    val metadataService = new JcrRepositoryMetadataService(template)
    if (null != repositoryId) {
      metadataService.validateAndStoreRepositoryId(repositoryId)
    }
  }
}