package com.xebialabs.deployit.repository;

import java.util.*;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.jcr.JcrQueryTemplate;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;

import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_NODETYPE_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME;
import static com.xebialabs.deployit.jcr.JcrConstants.LAST_MODIFIED_AT_PROPERTY_NAME;
import static java.lang.String.format;

public class SearchQueryBuilder {
    public static final String ESCAPE_CHARACTER = "\\";
    public static final String WILDCARD_CHARACTER = "%";
    static final char ESCAPE_CHAR = ESCAPE_CHARACTER.charAt(0);
    static final char WILDCARD_CHAR = WILDCARD_CHARACTER.charAt(0);

    public static final String CI_SELECTOR_NAME = "ci";

    private static final String EQUALITY_OPERATOR = "=";
    private static final String LIKE_OPERATOR = "LIKE";
    private static final String NAME_IS_OPERATOR = "NAME_IS";
    private static final String NAME_LIKE_OPERATOR = "NAME_LIKE";
    private static final String ISCHILDNODE_OPERATOR = "ISCHILDNODE";
    private static final String ISDESCENDANTNODE_OPERATOR = "ISDESCENDANTNODE";
    private static final String CONTAINS_OPERATOR = "CONTAINS";

    private final SearchParameters parameters;
    private final StringBuilder joins;
    private final List<Condition> conditions;
    private int nextSelectorId = 1;

    private List<Type> baseTypes = Arrays.asList(Type.valueOf(ConfigurationItem.class), Type.valueOf(BaseConfigurationItem.class));

    public SearchQueryBuilder(final SearchParameters parameters) {
        this.parameters = parameters;
        this.joins = new StringBuilder();
        this.conditions = new ArrayList<>();
    }

    protected JcrQueryTemplate createTemplate() {
        return createTemplate(new JcrQueryTemplateFactory());
    }

    protected JcrQueryTemplate createTemplate(JcrQueryTemplateFactory factory) {
        createJoinsAndConditions();

        String queryString = constructQueryString();
        JcrQueryTemplate template = factory.createTemplate(queryString);
        addParameters(template);
        template.setResultsPerPage(parameters.getResultsPerPage());
        template.setPage(parameters.getPage());
        template.setDepth(parameters.getDepth());

        return template;
    }

    private String constructQueryString() {
        StringBuilder builder = new StringBuilder();
        builder.append("SELECT " + CI_SELECTOR_NAME + ".* FROM [" + CONFIGURATION_ITEM_NODETYPE_NAME + "] AS " + CI_SELECTOR_NAME);
        builder.append(joins.toString());

        if (!conditions.isEmpty()) {
            builder.append(" WHERE ");
            builder.append(String.join(" AND ", conditions.stream().map(Condition::build).collect(Collectors.toList())));
        }

        builder.append(" ORDER BY NAME(" + CI_SELECTOR_NAME + ")");

        logger.trace("Query string built: {}", builder.toString());

        return builder.toString();
    }

    private void addParameters(JcrQueryTemplate query) {
        for (Condition condition : conditions) {
            if (condition.parameter == null)
                continue;

            for (int i = 0; i < condition.values.length; i++) {
                String key = condition.parameter + i;
                Object value = condition.values[i];

                query.setParameter(key, value);
                logger.trace("Bound {} to {}", key, value);
            }
        }
    }

    private void createJoinsAndConditions() {
        createConditionForParent();
        createConditionForAncestor();
        createConditionForName();
        createConditionForConfigurationItemTypeName();
        createConditionForDate();
        createConditionsForProperties();
    }

    private void createConditionForConfigurationItemTypeName() {
        if (parameters.getType() != null && !isBaseType(parameters.getType())) {
            List<String> types = new ArrayList<>();
            types.add(parameters.getType().toString());
            for (Type subtype : DescriptorRegistry.getSubtypes(parameters.getType())) {
                types.add(subtype.toString());
            }
            conditions.add(Condition.from(CI_SELECTOR_NAME, CONFIGURATION_ITEM_TYPE_PROPERTY_NAME, "_configurationItemTypeName", EQUALITY_OPERATOR, false,
                    (Object[]) types.toArray(new Object[types.size()])));
        }
    }

    private boolean isBaseType(Type type) {
        return baseTypes.contains(type);
    }

    private void createConditionForParent() {
        if (com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank(parameters.getParent())) {
            String parent = parameters.getParent();
            if (!parent.startsWith("/")) {
                parent = "/" + parent;
            }
            conditions.add(Condition.isChildNode(CI_SELECTOR_NAME, parent));
        }
    }

    private void createConditionForAncestor() {
        if (Strings.isNotBlank(parameters.getAncestor())) {
            String ancestor = parameters.getAncestor();
            if (!ancestor.startsWith("/")) {
                ancestor = "/" + ancestor;
            }
            conditions.add(Condition.isDescendantNode(CI_SELECTOR_NAME, ancestor));
        }
    }

    private void createConditionForName() {
        String name = parameters.getName();
        if (Strings.isNotBlank(name)) {
            if (parameters.isExactNameSearch()) {
                conditions.add(Condition.nameIs(CI_SELECTOR_NAME, parameters.getName()));
            } else {
                if (name.contains(WILDCARD_CHARACTER)) {
                    if (isWildCardBasedNameSearch(name)) {
                        conditions.add(Condition.nameLike(CI_SELECTOR_NAME, name));
                    } else {
                        // Un-escaping since case insensitive like search isn't supported by jcr
                        conditions.add(Condition.nameIs(CI_SELECTOR_NAME, unescapeWildCard(name)));
                    }

                } else {
                    conditions.add(Condition.nameIs(CI_SELECTOR_NAME, name));
                }
            }
        }
    }
    
    /**
     * Unescapes the wildcard character
     *
     * @param name
     * @return
     */
    static String unescapeWildCard(String name) {
        return name.replace(ESCAPE_CHARACTER + WILDCARD_CHARACTER, WILDCARD_CHARACTER);
    }

    static boolean isWildCardBasedNameSearch(String name) {
        for (int i = 0; i < name.length(); i++) {
            char ch = name.charAt(i);
            if (ch == WILDCARD_CHAR) {
                if (i == 0) {
                    return true;
                }
                if (name.charAt(i - 1) != ESCAPE_CHAR) {
                    return true;
                }
            }
        }
        return false;
    }    

    private void createConditionForDate() {
        createConditionForDate(parameters.getBefore(), "_before", "<=");
        createConditionForDate(parameters.getAfter(), "_after", ">=");
    }

    private void createConditionForDate(Calendar field, String parameter, String operator) {
        if (field != null) {
            conditions.add(Condition.from(CI_SELECTOR_NAME, LAST_MODIFIED_AT_PROPERTY_NAME, parameter, operator, field));
        }
    }

    private void createConditionsForProperties() {
        for (Map.Entry<String, String> entry : parameters.getProperties().entrySet()) {
            createConditionForProperty(entry.getKey(), entry.getValue());
        }
    }

    private void createConditionForProperty(final String propertyName, final String propertyValue) {
        conditions.add(Condition.match(CI_SELECTOR_NAME, propertyName, propertyName, propertyValue));
    }

    private static class Condition {
        String selector;
        String field;
        String operator;
        String parameter;
        Object[] values;
        boolean caseInsensitive;

        private static final String NAME_IS_FORMAT = "NAME(%s) = \'%s\'";
        private static final String NAME_LIKE_FORMAT = "LOWER(NAME(%s)) LIKE \'%s\'";
        private static final String ISCHILDNODE_FORMAT = ISCHILDNODE_OPERATOR + "(%s, [\'%s\'])";
        private static final String ISDESCENDANTNODE_FORMAT = ISDESCENDANTNODE_OPERATOR + "(%s, [\'%s\'])";
        private static final String CONTAINS_FORMAT = "contains(%s.[%s], $%s)";
        private static final String CASE_SENSITIVE = "%s.[%s] %s $%s";
        private static final String CASE_INSENSITIVE = "LOWER(%s.[%s]) %s $%s";

        String build() {
            StringBuilder conditionString = new StringBuilder();
            conditionString.append("(");
            for (int i = 0; i < values.length; i++) {
                if (i > 0) {
                    conditionString.append(" OR ");
                }

                if (operator.equals(NAME_IS_OPERATOR)) {
                    conditionString.append(format(NAME_IS_FORMAT, selector, escapeSingleQuote(values[i])));
                } else if (operator.equals(NAME_LIKE_OPERATOR)) {
                    conditionString.append(format(NAME_LIKE_FORMAT, selector, escapeSingleQuote(values[i])));
                } else if (operator.equals(ISCHILDNODE_OPERATOR)) {
                    conditionString.append(format(ISCHILDNODE_FORMAT, selector, escapeSingleQuote(values[i])));
                } else if (operator.equals(ISDESCENDANTNODE_OPERATOR)) {
                    conditionString.append(format(ISDESCENDANTNODE_FORMAT, selector, escapeSingleQuote(values[i])));
                } else if (operator.equals(CONTAINS_OPERATOR)) {
                    conditionString.append(format(CONTAINS_FORMAT, selector, field, parameter + i));
                } else {
                    conditionString.append(format(caseInsensitive ? CASE_INSENSITIVE : CASE_SENSITIVE, selector, field, operator, parameter + i));
                }
            }
            conditionString.append(")");
            return conditionString.toString();
        }

        private String escapeSingleQuote(Object value) {
            return ((String) value).replaceAll("'", "''");
        }

        static Condition match(final String selector, final String field, final String parameter, final String value) {
            String operator;
            boolean caseInsensitive;
            if (value.contains(WILDCARD_CHARACTER)) {
                operator = LIKE_OPERATOR;
                caseInsensitive = true;
            } else {
                operator = EQUALITY_OPERATOR;
                caseInsensitive = false;
            }
            return from(selector, field, parameter, operator, caseInsensitive, value);
        }

        static Condition nameIs(final String selector, final String value) {
            return from(selector, null, null, NAME_IS_OPERATOR, false, value);
        }

        static Condition nameLike(final String selector, final String value) {
            return from(selector, null, null, NAME_LIKE_OPERATOR, false, value.toLowerCase());
        }

        static Condition isChildNode(final String selector, final String value) {
            return from(selector, null, null, ISCHILDNODE_OPERATOR, false, value);
        }

        static Condition isDescendantNode(final String selector, final String value) {
            return from(selector, null, null, ISDESCENDANTNODE_OPERATOR, false, value);
        }

        static Condition textSearch(String property, String textQuery, String parameterName) {
            return Condition.from(CI_SELECTOR_NAME, property, parameterName, CONTAINS_OPERATOR, false, textQuery);
        }

        static Condition from(final String selector, final String field, final String parameter, final String operator, final Object value) {
            return from(selector, field, parameter, operator, false, value);
        }

        static Condition from(final String selector, final String field, final String parameter, final String operator, final boolean caseInsensitive,
                              final Object... values) {
            Condition t = new Condition();
            t.selector = selector;
            t.field = field;
            t.parameter = parameter;
            t.operator = operator;
            t.caseInsensitive = caseInsensitive;
            t.values = values;
            return t;
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(SearchQueryBuilder.class);
}
