package com.xebialabs.deployit.repository;

import com.xebialabs.deployit.engine.spi.event.*;
import com.xebialabs.deployit.event.EventBusHolder;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.jcr.*;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.license.service.LicenseService;
import com.xebialabs.xlplatform.repository.JcrLicenseCiCounterFactory;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.PostConstruct;
import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.query.Query;
import javax.jcr.query.QueryResult;
import javax.jcr.query.RowIterator;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.deployit.repository.PathHelper.getAbsolutePathFromId;
import static com.xebialabs.deployit.repository.SearchQueryBuilder.CI_SELECTOR_NAME;

public class JcrRepositoryService implements RepositoryService {

    private final JcrTemplate jcrTemplate;
    private final PasswordEncrypter passwordEncrypter;

    private LicenseService licenseService;

    private boolean reuseNodeReaderContext = true;

    public JcrRepositoryService(JcrTemplate jcrTemplate, PasswordEncrypter passwordEncrypter, LicenseService licenseService) {
        this.jcrTemplate = jcrTemplate;
        this.passwordEncrypter = passwordEncrypter;
        this.licenseService = licenseService;
        new JcrTemplateHolder(jcrTemplate);
    }

    @PostConstruct
    public void initializeLicenseValidator() {
        licenseService.initialize(new JcrLicenseCiCounterFactory(this));
    }

    @Override
    public boolean exists(final String id) {
        logger.debug("Checking whether node [{}] exists.", id);

        return jcrTemplate.execute(session -> session.itemExists(getAbsolutePathFromId(id)));
    }

    @Override
    public <T extends ConfigurationItem> T read(final String id) {
        return read(id, Integer.MAX_VALUE, null, reuseNodeReaderContext);
    }

    @Override
    public <T extends ConfigurationItem> T read(final String id, int depth) {
        return read(id, depth, null, reuseNodeReaderContext);
    }

    @Override
    public <T extends ConfigurationItem> T read(String id, WorkDir workDir) {
        return read(id, Integer.MAX_VALUE, workDir, true);
    }

    @Override
    public <T extends ConfigurationItem> T read(final String id, final boolean useCache) {
        return read(id, Integer.MAX_VALUE, null, useCache);
    }

    @Override
    public <T extends ConfigurationItem> T read(final String id, final WorkDir workDir, final boolean useCache) {
        return read(id, Integer.MAX_VALUE, workDir, useCache);
    }

    @Override
    public <T extends ConfigurationItem> List<T> read(final List<String> ids, final int depth, final boolean useCache) {
        return ids.stream().map(id -> this.<T>read(id, useCache)).collect(Collectors.toList());
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T extends ConfigurationItem> T read(final String id, final int depth, final WorkDir workDir, final boolean useCache) {
        logger.debug("Reading node [{}] with depth {} and workdir {} (useCache={})", id, depth, workDir, useCache);
        checkNotNull(id, "id is null");

        return (T) _read(id, depth, workDir, useCache, true);
    }

    private <T extends ConfigurationItem> T _read(final String id, final int depth, final WorkDir workDir, final boolean useCache, final boolean failOnMissing) {
        return jcrTemplate.execute(session -> {
            String absolutePathFromId = getAbsolutePathFromId(id);
            boolean exists = session.nodeExists(absolutePathFromId);
            if (failOnMissing && !exists) {
                throw new NotFoundException("Repository entity [%s] not found", id);
            } else if (!exists) {
                return null;
            }

            Node node = session.getNode(absolutePathFromId);
            if (useCache) {
                return NodeReader.<T>read(session, node, depth, workDir, passwordEncrypter);
            } else {
                return NodeReader.<T>read(session, node, depth, workDir, new NodeReaderContext(), passwordEncrypter);
            }
        });
    }

    @Override
    public List<ConfigurationItemData> list(SearchParameters parameters) {
        checkNotNull(parameters, "parameters is null");
        logger.debug("Listing node IDs with parameters {}.", parameters);

        JcrQueryTemplate queryTemplate = new SearchQueryBuilder(parameters).createTemplate();
        return list(queryTemplate);
    }

    public <T extends ConfigurationItem> List<T> listEntities(SearchParameters parameters) {
        checkNotNull(parameters, "parameters is null");
        logger.debug("Listing nodes with parameters {}.", parameters);

        JcrQueryTemplate queryTemplate = new SearchQueryBuilder(parameters).createTemplate();
        return listEntities(queryTemplate);
    }

    public List<ConfigurationItemData> list(JcrQueryTemplate queryTemplate) {
        checkNotNull(queryTemplate, "queryTemplate is null");
        logger.debug("Listing node IDs with JCR query template {}.", queryTemplate);

        // Setting explicit depth=0 as no depth is need
        queryTemplate.setDepth(0);

        List<ConfigurationItemData> entities = query(queryTemplate, (ResultSetExtractor<ConfigurationItemData, JcrResult>) result -> {
            try {
                Node node = result.getRow().getNode(CI_SELECTOR_NAME);
                Type type = NodeReader.readType(node);
                if (type != null && type.exists()) {
                    return new ConfigurationItemData(PathHelper.getIdFromAbsolutePath(node.getPath()), type);
                } else {
                    return null;
                }
            } catch (RepositoryException re) {
                logger.error("Error while listing Configuration Item ids", re);
                throw new RuntimeRepositoryException(re, "Error while listing ConfigurationItem ids for '%s'", queryTemplate);
            }
        });
        return entities.stream().map((ci) -> new ConfigurationItemData(ci.getId(), ci.getType())).collect(Collectors.toList());
    }

    protected  <T extends ConfigurationItem> List<T> listEntities(final JcrQueryTemplate queryTemplate) {
        checkNotNull(queryTemplate, "query is null");
        logger.debug("Listing nodes with JCR query template {}.", queryTemplate);

        return jcrTemplate.execute((JcrCallback<List<T>>) session -> {
            final Query query = queryTemplate.createQuery(session);
            final QueryResult queryResult = query.execute();
            return getOrderedEntities(session, queryResult, queryTemplate.getDepth());
        });
    }

    protected  <T, R extends Result> List<T> query(JcrQueryTemplate queryTemplate, ResultSetExtractor<T, R> extractor) {
        checkNotNull(queryTemplate, "query is null");
        logger.debug("Querying with JCR query template {}.", queryTemplate);

        return jcrTemplate.execute(session -> {
            final Query query = queryTemplate.createQuery(session);
            final QueryResult queryResult = query.execute();

            final List<T> items = new ArrayList<>();
            final RowIterator iterator = queryResult.getRows();
            while (iterator.hasNext()) {
                @SuppressWarnings("unchecked")
                R result = (R) new JcrResult(iterator.nextRow());
                T t = extractor.extractData(result);
                if (t != null) {
                    items.add(t);
                }
            }
            return items;
        });
    }

    private <T extends ConfigurationItem> List<T> getOrderedEntities(Session session, QueryResult queryResult, int depth) throws RepositoryException {
        final List<T> items = new ArrayList<>();
        final RowIterator iterator = queryResult.getRows();
        while (iterator.hasNext()) {
            Node node = iterator.nextRow().getNode(CI_SELECTOR_NAME);
            // Your IDE might not think <T> is needed here, but your compiler does ;)
            T item = NodeReader.<T>read(session, node, depth, null, passwordEncrypter);
            items.add(item);
        }
        return items;
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T extends ConfigurationItem> void create(final T... cis) {
        ChangeSet changeset = new ChangeSet();
        changeset.create(Arrays.asList(cis));
        execute(changeset);

        EventBusHolder.publish(new CisCreatedEvent(cis));
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T extends ConfigurationItem> void update(final T... cis) {
        ChangeSet changeset = new ChangeSet();
        changeset.update(Arrays.asList(cis));
        execute(changeset);

        EventBusHolder.publish(new CisUpdatedEvent(changeset.getUpdateCis()));
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T extends ConfigurationItem> void createOrUpdate(T... cis) {
        ChangeSet changeSet = new ChangeSet();
        changeSet.createOrUpdate(Arrays.asList(cis));
        execute(changeSet);

        EventBusHolder.publish(new CisCreatedEvent(new ArrayList<>(changeSet.getCreateOrUpdateActualCreatedCis())));
        EventBusHolder.publish(new CisUpdatedEvent(new ArrayList<>(changeSet.getCreateOrUpdateActualUpdatedCis())));
    }

    @Override
    public void delete(final String... ids) {
        ChangeSet changeset = new ChangeSet();
        // Read for auditing
        List<ConfigurationItem> cis = new ArrayList<>();
        for (String id : ids) {
            ConfigurationItem e = _read(id, 0, null, reuseNodeReaderContext, false);
            if (e != null) {
                cis.add(e);
            }
        }

        changeset.delete(Arrays.asList(ids));
        checkReferentialIntegrity(changeset);
        execute(changeset);


        EventBusHolder.publish(new CisDeletedEvent(cis));
    }

    @Override
    public void rename(final String id, final String newName) {
        checkArgument(newName.indexOf('/') == -1, "New name [%s] should not contain a /", newName);
        ConfigurationItem configurationItem = _read(id, 0, null, reuseNodeReaderContext, true);
        ChangeSet changeSet = new ChangeSet();
        changeSet.rename(id, newName);
        execute(changeSet);

        EventBusHolder.publish(new CiRenamedEvent(configurationItem, newName));
    }

    @Override
    public <T extends ConfigurationItem> void move(final String id, final String newId) {
        logger.debug("Moving node [{}] to [{}]", id, newId);

        ConfigurationItem configurationItem = _read(id, 0, null, reuseNodeReaderContext, true);
        ChangeSet changeSet = new ChangeSet();
        changeSet.move(id, newId);
        execute(changeSet);


        EventBusHolder.publish(new CiMovedEvent(configurationItem, newId));
    }

    @Override
    public void copy(final String id, final String newId) {
        ChangeSet changeSet = new ChangeSet();
        changeSet.copy(id, newId);
        execute(changeSet);
        ConfigurationItem configurationItem = _read(id, 0, null, reuseNodeReaderContext, true);

        EventBusHolder.publish(new CiCopiedEvent(configurationItem, newId));
    }

    /**
     * Checks if a type may be moved. Defaults is no-op. Subclasses may override this method and throw a
     * {@link RuntimeException} if they want to prevent the type from being moved.
     *
     * @param type
     *            the type that is to be moved
     * @throws RuntimeException
     *             if the move is not allowed on the type.
     */
    protected void checkMoveAllowed(Type type) {
    }

    /**
     * Checks if a type may be copied. Defaults is no-op. Subclasses may override this method and throw a
     * {@link RuntimeException} or {@link javax.jcr.RepositoryException} if they want to prevent the type from being moved.
     *
     * @param session a JCR session so you can access the repository to do the checking
     * @param toBeCopied the path of the CI that will be copied
     * @param newId the destination path of the CI
     *
     * @throws javax.jcr.RepositoryException
     *             if the copy is not allowed on the type.
     * @throws RuntimeException
     *             if the copy is not allowed on the type.
     */
    protected void checkCopyAllowed(Session session, String toBeCopied, String newId) throws RepositoryException {
    }


    static Type readType(Session session, String id) throws RepositoryException {
        return Type.valueOf(session.getNode(getAbsolutePathFromId(id)).getProperty(JcrConstants.CONFIGURATION_ITEM_TYPE_PROPERTY_NAME).getString());
    }

    static String extractParentId(String id) {
        int indexOf = id.lastIndexOf('/');
        if (indexOf != -1) {
            return id.substring(0,indexOf);
        }
        return id;
    }

    @Override
    public void execute(ChangeSet changeset) {
        execute(changeset, new NullProgressLogger());
    }

    @Override
    public void execute(ChangeSet changeset, ProgressLogger progressLogger) {
        jcrTemplate.execute(new JcrChangeSetExecutor(this, changeset, passwordEncrypter, licenseService , progressLogger));
    }

    @Override
    public void checkReferentialIntegrity(final ChangeSet changeset) throws ItemInUseException, ItemAlreadyExistsException {
        jcrTemplate.execute(new JcrReferentialIntegrityChecker(this, changeset, passwordEncrypter, licenseService));
    }

    public void setReuseNodeReaderContext(boolean reuseNodeReaderContext) {
        this.reuseNodeReaderContext = reuseNodeReaderContext;
    }

    private static Logger logger = LoggerFactory.getLogger(JcrRepositoryService.class);

}
